/**
 *******************************************************************************
 * @file    tmpm311_temp.c
 * @brief   This file provides API functions for TEMP driver.
 * @version V2.0.2.1
 * @date    2014/09/04
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * (C)Copyright TOSHIBA ELECTRONIC DEVICES & STORAGE CORPORATION 2017 All rights reserved
 *******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "tmpm311_temp.h"


#if defined(__TMPM311_TEMP_H)
/** @addtogroup TX03_Periph_Driver
  * @{
  */

/** @defgroup TEMP 
  * @brief TEMP driver modules
  * @{
  */
/** @defgroup TEMP_Private_Defines
  * @{
  */

#define TEMP_EN_EN1_SET         ((uint32_t)0x00000002)
#define TEMP_EN_EN1_CLEAR       ((uint32_t)0xFFFFFFFD)
#define TEMP_EN_EN0_SET         ((uint32_t)0x00000001)
#define TEMP_EN_EN0_CLEAR       ((uint32_t)0xFFFFFFFE)

#define TEMP_CR_CR0_SET         ((uint32_t)0x00000001)
#define TEMP_CR_CR0_CLEAR       ((uint32_t)0xFFFFFFFE)

/** @} */
/* End of group TEMP_Private_Defines */

/** @defgroup TEMP_Private_FunctionPrototypes
  * @{
  */

/** @} */
/* End of group TEMP_Private_FunctionPrototypes */

/** @defgroup TEMP_Private_Functions
  * @{
  */

/** @} */
/* End of group group TEMP_Private_Functions */

/** @defgroup TEMP_Exported_Functions
  * @{
  */

/**
  * @brief  Enable or disable AMP operation for ADC.
  * @param  NewState: Specify the AMP operation.
  *   This parameter can be one of the following values:
  *   ENABLE, DISABLE.
  * @retval None
  * @note  The AMP operation must be enabled when BGR operation is enabled.
  * @register The used register:
  *   TEMPEN<EN1>
  */
void TEMP_SetAMPState(FunctionalState NewState)
{
    /* Check the parameters */
    assert_param(IS_FUNCTIONAL_STATE(NewState));

    if (NewState == ENABLE) {
        TSB_TEMP->EN |= TEMP_EN_EN1_SET;
    } else {
        TSB_TEMP->EN &= TEMP_EN_EN1_CLEAR;
    }
}

/**
  * @brief  Get AMP operation state for ADC.
  * @param  None
  * @retval The AMP operation state.
  *   The value returned can be one of the following values:
  *   ENABLE, DISABLE.
  * @register The used register:
  *   TEMPEN<EN1>
  */
FunctionalState TEMP_GetAMPState(void)
{
    uint32_t en = TSB_TEMP->EN & TEMP_EN_EN1_SET;
    FunctionalState retval = DISABLE;

    if (en == TEMP_EN_EN1_SET) {
        retval = ENABLE;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Enable or disable the reference voltage circuit.
  * @param  NewState: Specify the reference voltage circuit.
  *   This parameter can be one of the following values:
  *   ENABLE, DISABLE.
  * @retval None
  * @register The used register:
  *   TEMPEN<EN0>
  */
void TEMP_SetBGRState(FunctionalState NewState)
{
    /* Check the parameters */
    assert_param(IS_FUNCTIONAL_STATE(NewState));

    if (NewState == ENABLE) {
        TSB_TEMP->EN |= TEMP_EN_EN0_SET;
    } else {
        TSB_TEMP->EN &= TEMP_EN_EN0_CLEAR;
    }
}

/**
  * @brief  Get the reference voltage circuit state.
  * @param  None
  * @retval The reference voltage circuit state.
  *   The value returned can be one of the following values:
  *   ENABLE, DISABLE.
  * @register The used register:
  *   TEMPEN<EN0>
  */
FunctionalState TEMP_GetBGRState(void)
{
    uint32_t en = TSB_TEMP->EN & TEMP_EN_EN0_SET;
    FunctionalState retval = DISABLE;

    if (en == TEMP_EN_EN0_SET) {
        retval = ENABLE;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Enable or disable temperature sensor operation.
  * @param  NewState: Specify the temperature sensor operation.
  *   This parameter can be one of the following values:
  *   ENABLE, DISABLE.
  * @retval None
  * @register The used register:
  *   TEMPCR<CR0>
  */
void TEMP_SetSensorState(FunctionalState NewState)
{
    /* Check the parameters */
    assert_param(IS_FUNCTIONAL_STATE(NewState));

    if (NewState == ENABLE) {
        TSB_TEMP->CR |= TEMP_CR_CR0_SET;
    } else {
        TSB_TEMP->CR &= TEMP_CR_CR0_CLEAR;
    }
}

/**
  * @brief  Get the temperature sensor operation state.
  * @param  None
  * @retval The temperature sensor operation state.
  *   The value returned can be one of the following values:
  *   ENABLE, DISABLE.
  * @register The used register:
  *   TEMPCR<CR0>
  */
FunctionalState TEMP_GetSensorState(void)
{
    uint32_t cr = TSB_TEMP->CR & TEMP_CR_CR0_SET;
    FunctionalState retval = DISABLE;

    if (cr == TEMP_CR_CR0_SET) {
        retval = ENABLE;
    } else {
        /* Do nothing */
    }

    return retval;
}

/** @} */
/* End of group TEMP_Exported_Functions */

/** @} */
/* End of group TEMP */

/** @} */
/* End of group TX03_Periph_Driver */
#endif                          /* defined(__TMPM311_TEMP_H) */
