/** 
 *******************************************************************************
 * @file    retarget.c
 * @brief   the application functions of UART_Retarget for the TOSHIBA
 *          'TMPM311' Device Series 
 * @version V0.100
 * @date    2014/10/24
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * (C)Copyright TOSHIBA ELECTRONIC DEVICES & STORAGE CORPORATION 2017 All rights reserved
 *******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/

#include "retarget.h"
/* External function prototypes ----------------------------------------------*/

//#define BUFFER_SIZE 16U
//#define SET     0x01U           /* flag is set */
//#define CLEAR   0x00U           /* flag is cleared */
//#define NO      0x00U           /* Send finish NG */
//#define YES     0x01U           /* Send finish OK */


uint8_t gSIOTxBuffer[BUFFER_SIZE] = { 0 };

uint8_t gSIORdIndex = 0U;
uint8_t gSIOWrIndex = 0U;
uint8_t fSIO_INT = 0U;
uint8_t fSIOTxOK = NO;

extern uint8_t send_char(uint8_t ch);


/**
  * @brief  Retarget printf to uart.
  * @param  IAR:Character to be sent.KEIL:Character to be sent.
  * @retval Character to be sent
  */

#if defined ( __CC_ARM   )      /* RealView Compiler */
struct __FILE {
    int handle;                 /* Add whatever you need here */
};
FILE __stdout;
FILE __stdin;
int fputc(int ch, FILE * f)
#elif defined ( __ICCARM__ )    /*IAR Compiler */
int putchar(int ch)
#endif
{
    return (send_char(ch));
}

/**
  * @brief  SIO Configuration.
  * @param  SCx
  * @retval None
  */
void SIO_Configuration( )
{

    TSB_RETG->FR1 |= RXD_BIT|TXD_BIT;
    TSB_RETG->CR |=  TXD_BIT;
    TSB_RETG->IE |= RXD_BIT;
    TSB_RETG->PUP |= RXD_BIT|TXD_BIT;  
}


/**
  * @brief  SIO Configuration Initialize the uart port (SIO2).
  * @param  None
  * @retval None
  */
void SIO_ChInit(TSB_SC_TypeDef * SCx)
{
    UART_InitTypeDef myUART;

    /* configure SIO0 for reception */
    UART_Enable(SCx);
    myUART.BaudRate = 115200U;   /* baud rate = 115200 */
    myUART.DataBits = UART_DATA_BITS_8; /* no handshake, 8-bit data, clock by baud rate generator */
    myUART.StopBits = UART_STOP_BITS_1; /* 1-bit stop, LSB,  W-buff enable */
    myUART.Parity = UART_NO_PARITY;
    myUART.Mode = UART_ENABLE_TX;
    myUART.FlowCtrl = UART_NONE_FLOW_CTRL;
    UART_Init(SCx, &myUART);
}

/**
  * @brief  Initialize Retarget.
  * @param  None
  * @retval None
  */
  
void Retarget_Init()
{
    SIO_Configuration();
    SIO_ChInit(UART_RETARGET);
}

 
/**
  * @brief  Send character by SIO0
  * @param  Character to be sent
  * @retval Character to be sent
  */
uint8_t send_char(uint8_t ch)
{
    while (gSIORdIndex != gSIOWrIndex) {        /* wait for finishing sending */
        /* do nothing */
    }
    gSIOTxBuffer[gSIOWrIndex++] = ch;   /* fill TxBuffer */
    if (fSIO_INT == CLEAR) {    /* if SIO INT disable, enable it */
        fSIO_INT = SET;         /* set SIO INT flag */
        UART_SetTxData(UART_RETARGET, gSIOTxBuffer[gSIORdIndex++]);
        NVIC_EnableIRQ(INT_RETARGET_IRQn);
    }

    return ch;
}

/**
  * @brief  INTTXx Interrupt Service Routine.
  * @param  None
  * @retval None
  */

void Retartget_IRQHandler(void)
{
    if (gSIORdIndex < gSIOWrIndex) {    /* buffer is not empty */
        UART_SetTxData(UART_RETARGET, gSIOTxBuffer[gSIORdIndex++]);     /* send data */
        fSIO_INT = SET;         /* SIOx INT is enable */
    } else {
        /* disable SIOx INT */
        fSIO_INT = CLEAR;
        NVIC_DisableIRQ(INT_RETARGET_IRQn);
        fSIOTxOK = YES;
    }
    if (gSIORdIndex >= gSIOWrIndex) {   /* reset buffer index */
        gSIOWrIndex = CLEAR;
        gSIORdIndex = CLEAR;
    } else {
        /* do nothing */
    }
}
