/**
 ****************************************************************************
 * @file     E_Sub.c
 * @brief    Sub Routine Functions
 * @version  V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2021
 *****************************************************************************
 */
#include "ipdefine.h"

#define DEFINE_APP
#include "E_Sub.h"
#undef DEFINE_APP


/**
 *********************************************************************************************
 * @brief		Cal Sine
 *				sin(x) = 3.140625x + 0.02026367x^2 -5.325196x^3 + 0.5446778x^4 + 1.800293x^5
 *
 * @param		short	theta	theta
 *
 * @return		short	sin(theta)
 *********************************************************************************************
 */
#define cSine1		(short)(FIX_12 *  3.140625)
#define cSine2		(short)(FIX_12 *  0.02026367)
#define cSine3		(short)(FIX_12 * -5.325196)
#define cSine4		(short)(FIX_12 *  0.5446778)
#define cSine5		(short)(FIX_12 *  1.800293)

short E_Sine(short theta)
{
	short x;
	short x1;
	long temp32;

	x = (short)(theta & 0x7fff);
	if (x >= ELE_DEG(90))
	{
		x = (q15_t)(ELE_DEG(180) - x);
	}
	x1 = x;
	temp32 = (cSine1 * (long)x1) << 1;
	x1 = (short)(x1 * (long)x >> (16 - 1));
	temp32 += (cSine2 * (long)x1) << 1;
	x1 = (short)(x1 * (long)x >> (16 - 1));
	temp32 += (cSine3 * (long)x1) << 1;
	x1 = (short)(x1 * (long)x >> (16 - 1));
	temp32 += (cSine4 * (long)x1) << 1;
	x1 = (short)(x1 * (long)x >> (16 - 1));
	temp32 += (cSine5 * (long)x1) << 1;
	temp32 <<= 3;

	if ((unsigned long)temp32 >= 0x7fffffff)
	{
		temp32 = 0x7fffffff;
	}
	if (theta <= 0)
	{
		temp32 *= ( -1);
	}

	return ((short)(temp32 >> 16));
}


/**
 *********************************************************************************************
 * @brief		E_Cosine
 *
 * @param		short	theta
 *
 * @return		short
 *********************************************************************************************
 */
short E_Cosine(short theta)
{
	return (E_Sine((short)(ELE_DEG(90) + theta)));
}

/**
 *********************************************************************************************
 * @brief		E_Clarke
 *
 * @param		q15_t 		_iu			U-phase current
 * @param		q15_t 		_iv			V-phase current
 * @param		q15_t 		_iw			W-phase current
 * @param		q15_t* 		_ialpha		alpha-axis current storage address
 * @param		q15_t* 		_ibeta		beta-axis current storage address
 *
 * @return		none	
 *********************************************************************************************
 */
void E_Clarke(q15_t _iu, q15_t _iv, q15_t _iw, q15_t* _ialpha, q15_t* _ibeta)
{

	int32_t temp_1;

	/* Ialpha=Iu */
	*_ialpha = _iu;
	/* Ibeta=(Iu + 2*Iv)/sqrt3 */
	temp_1 = ((int32_t)_iu  + (int32_t)(_iv * 2));
	*_ibeta = (temp_1 * (int32_t)((1L << 15) * (1.0f / 1.7320508f))) >> 15;
}

/**
 *********************************************************************************************
 * @brief		E_Park
 *
 * @param		q15_t 		_ialpha		alpha-axis 
 * @param		q15_t		_ibeta		bata-axis
 * @param		uint16_t	theta		Theta(position)
 * @param		q15_t* 		_id			d-axis storage address
 * @param		q15_t* 		_iq			q-axis storage address
 *
 * @return		none	
 *********************************************************************************************
 */
void E_Park(q15_t _ialpha, q15_t _ibeta, uint16_t _theta, q15_t* _id, q15_t* _iq)
{
	int32_t temp_1, temp_2, temp_3, temp_4;
	int16_t temp_sin, temp_cos;

	/*alpha_beta/dq conversion (Park.t) */
	temp_cos = E_Cosine(_theta);
	temp_sin = E_Sine(_theta);

	/* Id = Ialpha * cos(theta)+Ibeta * sin(theta) */
    temp_1 = ((int32_t)_ialpha * temp_cos) << 1;
    temp_2 = ((int32_t)_ibeta  * temp_sin) << 1;
	*_id = (q15_t)(__QADD(temp_1, temp_2) >> 16);

	/* Iq = Ialpha * -sin(theta) + Ibeta * cos(theta) */
	temp_3 = ((int32_t)_ialpha * (int32_t)(-temp_sin)) << 1;
    temp_4 = ((int32_t)_ibeta  * (int32_t)temp_cos) << 1;
	*_iq = (q15_t)(__QADD(temp_3, temp_4) >> 16);
}

/**
 *********************************************************************************************
 * @brief		E_InvPark
 *
 * @param		q31_t 		_d			d-axis
 * @param		q31_t 		_q			q-axis
 * @param		uint16_t	theta		Theta(position)
 * @param		q31_t* 		_alpha		alpha-axis storage address
 * @param		q31_t* 		_beta		bata-axis storage address
 *
 * @return		none	
 *********************************************************************************************
 */
void E_InvPark(q31_t _d, q31_t _q, uint16_t _theta, q31_t* _alpha, q31_t* _beta)
{
	int32_t temp_1, temp_2, temp_3, temp_4;
	int16_t temp_sin, temp_cos;

	temp_cos = E_Cosine(_theta);
	temp_sin = E_Sine(_theta);

	/* Valpha = Vd * cos(theta) - Vq * sin(theta) */
    temp_1 = (int32_t)(((int64_t)(_d) * (temp_cos)) >> 15);
    temp_2 = (int32_t)(((int64_t)(_q) * (temp_sin)) >> 15);
    *_alpha = __QSUB(temp_1, temp_2);

	/* Vbeta = Vd * sin(theta) + Vq * cos(theta) */
	temp_3 = (int32_t)(((int64_t)(_d) * (temp_sin)) >> 15);
    temp_4 = (int32_t)(((int64_t)(_q) * (temp_cos)) >> 15);
    *_beta = __QADD(temp_3, temp_4);
}

/**
 *********************************************************************************************
 * @brief		Get the maximum value of three data.
 *
 * @param		int32_t _a			Data a.
 * @param		int32_t _b			Data b.
 * @param		int32_t _c			Data c.
 *
 * @return		Max value
 *********************************************************************************************
 */
int32_t E_Max3(int32_t _a, int32_t _b, int32_t _c)
{
	int32_t	ret;

	if (_a >= _b)
	{
		ret = (_a < _c) ? _c : _a;
	}
	else
	{
		ret = (_b < _c) ? _c : _b;
	}
	return (ret);
}

/**
 *********************************************************************************************
 * @brief		Get the medium value of three data.
 *
 * @param		int32_t _a			Data a.
 * @param		int32_t _b			Data b.
 * @param		int32_t _c			Data c.
 *
 * @return		Medium value
 *********************************************************************************************
 */
int32_t E_Med3(int32_t _a, int32_t _b, int32_t _c)
{
	int32_t	ret;
	
	if (_a >= _b)
	{
		ret = (_c < _b) ? _b : (_c > _a) ? _a : _c;
	}
	else
	{
		ret = (_c < _a) ? _a : (_c > _b) ? _b : _c;
	}
	return (ret);
}

/**
 *********************************************************************************************
 * @brief		Get the minimum value of three data.
 *
 * @param		int32_t _a			Data a.
 * @param		int32_t _b			Data b.
 * @param		int32_t _c			Data c.
 *
 * @return		Minimum value
 *********************************************************************************************
 */
int32_t E_Min3(int32_t _a, int32_t _b, int32_t _c)
{
	int32_t	ret;

	if (_a >= _b)
	{
		ret = (_c < _b) ? _c : _b;
	}
	else
	{
		ret = (_c < _a) ? _c : _a;
	}
	return (ret);
}

/**
 ********************************************************************************************
 * @brief		E_Sort3.
 *
 * @param[in,out]	int32_t*	_n1	.
 * @param[in,out]	int32_t*	_n2	.
 * @param[in,out]	int32_t*	_n3	.
 *
 * @return		none
 ********************************************************************************************
 */
void E_Sort3(int32_t* _n1, int32_t* _n2, int32_t* _n3)
{
	int32_t	tmp;

	if (*_n1 > *_n2)
	{
		tmp = *_n1;
		*_n1 = *_n2;
		*_n2 = tmp;
	}
	if (*_n2 > *_n3)
	{
		tmp = *_n2;
		*_n2 = *_n3;
		*_n3 = tmp;
	}
	
	if (*_n1 > *_n2)
    {
		tmp = *_n1;
		*_n1 = *_n2;
		*_n2 = tmp;
	}
}

/*********************************** END OF FILE ******************************/
