// ****************************************************************************
// * @file     RD239-USB_I2C_Converter-01.ino
// * @brief    USB-I2C conversion of Arduino Nano Every for RD239 Reference Design
// * @version  V1.0
// *
// * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
// * 
// * Copyright(C) Toshiba Electronic Devices & Storage Corporation
// *****************************************************************************


// This firmware is used to recieve data from PC then send it to MCD
// or read data from MCD then send back to PC
//************************ Libraries ********************************
#include <Wire.h>


//********************* Global Variables ****************************
boolean debug = false;       // false:disable_debug_messages true:enable_debug_messages
int serialBaudrate = 9600;  // set UART baudrate
byte slaveId = 0x29;        // default Slave ID is 0x29 
byte commResultI2C = 0;     // I2C communication result // 0:success, 1:tooLongData, 2:NACKonAddress, 3:NACKonData, 4:other, 5:timeout


//******************* Function Declarations *************************
void writeI2Cbyte (byte regAdd, byte regData);  // Write data to MCD via I2C
byte readI2Cbyte(byte regAdd);  //reads a byte from I2C slave //slaveAdd, regAdd //returns read Byte
int readSerial(char result[]);  // Wait until command from PC and read serial data in cmdBuff until '\n'


//************************* Functions ********************************
void setup() {
  // put your setup code here, to run once:
  Serial.begin(serialBaudrate);
  Wire.begin(); // join i2c bus (address optional for master)
  pinMode(LED_BUILTIN,OUTPUT);
  identificationViaLed(); //to identify USB-I2C converter after powering ON, its LED blinks 2 times.
}

void loop() {
  byte cmdBuff[10];    // Stores the command received from Serial Port (PC)
  readSerial(cmdBuff); // Wait until command from PC and read serial data in cmdBuff until '\n'
  
  byte SR_ADDR, SR_DATA;
  switch (cmdBuff[0]) {  //cmdBuff[0] is command
    case 'd':   {         // checks if connected COM port device is USB-I2C converter or not
      //Return result
      Serial.print('#');             // Start of result
      Serial.print('d');             // d: connected device is USB-I2C converter
      Serial.print('\n');            //'\n' indicates end of result
      if (debug) {
        delay(60);
        Serial.println("Connected device is USB-I2C converter");
      }
    }
    break;
    case 'i':   {         // set slave ID
        slaveId = cmdBuff[1]; // I2C slave ID
        //Return result
        Serial.print('#');             // Start of result
        Serial.print('i');             // i: slaveId setting success
        Serial.print('\n');            //'\n' indicates end of result
        if (debug) {
          Serial.print("I2C_SlaveID = 0x");
          Serial.println(slaveId,HEX);
        }
      }
      break;
    case 'w':   {         // read data from PC then send to MCD  
        SR_ADDR = cmdBuff[1]; // I2C register address
        SR_DATA = cmdBuff[2]; // I2C data
        writeI2Cbyte(SR_ADDR, SR_DATA); //Write I2C data to register and updates commResultI2C
        //Return result
        Serial.print('#');             // Start of result 
        Serial.write(commResultI2C); // Returns I2C communication result  //'\n' indicates end of result
                                       // 0:success, 1:tooLongData, 2:NACKonAddress, 3:NACKonData, 4:other, 5:timeout
        Serial.print('\n');            //'\n' indicates end of result
        if(debug) {
          Serial.print("I2C_Write reg: 0x");
          Serial.print(SR_ADDR, HEX);
          Serial.print(" data: 0x");
          Serial.println(SR_DATA, HEX);
          if(commResultI2C != 0){
            Serial.println("I2C FAILURE. Check connection");
          }
        }
      }
      break;
    case 'r': {          // read data from MCD then send to PC  
        SR_ADDR = cmdBuff[1]; // I2C register address
        SR_DATA = readI2Cbyte(SR_ADDR);   //Reads I2C data and updates commResultI2C
        //Return result
        Serial.print('#');             // Start of result
        Serial.write(commResultI2C);   // Returns I2C communication result  
                                       // 0:success, 1:tooLongData, 2:NACKonAddress, 3:NACKonData, 4:other, 5:timeout
        Serial.write(SR_DATA);    // Returns read data
        Serial.print('\n');            //'\n' indicates end of result
        if(debug) {
          Serial.print("I2C_Read reg: 0x");
          Serial.print(SR_ADDR, HEX);
          Serial.print(" data: 0x");
          Serial.println(SR_DATA, HEX);
          if(commResultI2C != 0){
            Serial.println("I2C FAILURE. Check connection");
          }
        }
      }
      break;
    default:{                             //INVALID serial command
      //Return result 
      Serial.print('!');                  //: Invalid serial command
      Serial.print(" USB-I2C Converter: Invalid input command: ");
      Serial.print(cmdBuff[0]);
      }
      break;
  }
  if(commResultI2C!=0){ // I2C communication error
    digitalWrite(LED_BUILTIN,HIGH);
  }else{
    digitalWrite(LED_BUILTIN,LOW);
  }
}

void writeI2Cbyte (byte regAdd, byte regData) {   // Write data to MCD via I2C
  Wire.beginTransmission(slaveId);           // transmit to salve device
  Wire.write(regAdd);                             // set the addr
  Wire.write(regData);                            // send 8 bit data
  commResultI2C = Wire.endTransmission();         // stop transmitting // 0:success, 1:tooLongData, 2:NACKonAddress, 3:NACKonData, 4:other, 5:timeout
}

byte readI2Cbyte(byte regAdd){            //reads a byte from I2C slave //slaveAdd, regAdd //returns read Byte
  byte receivedData;
  Wire.beginTransmission(slaveId);   //Slave address + Write
  Wire.write(regAdd);                     //register address
  Wire.endTransmission(false);            //Re-start
  Wire.requestFrom((int)slaveId,1);  //slave address, qty, stop
  for(int i=0;i<10;i++){
    if(Wire.available()){
      receivedData=Wire.read();           //read data
      commResultI2C = 0;                  // Success
      break;
    }else{
      receivedData = 0;                   //Returning 0 because data cannot be read
      commResultI2C = 4;                  //Error data not received.
    }
    delay(2);
  }
  return receivedData;
}

int readSerial(char result[]) {  // Wait until command from PC and read serial data in cmdBuff until '\n'
  int i = 0;
  while (1) {
    while (Serial.available() > 0) {
      char inChar = Serial.read();
      if (i>2 && inChar == '\n') {     // Detects end of input command '\n'
        result[i] = '\0';
        Serial.flush();
        return 0;
      }
        result[i] = inChar;
        i++;
    }
  }
}

void identificationViaLed(){
  for(int i=0;i<2;i++){  //to identify USB-I2C converter after powering ON, it's blinks 2 times.
    digitalWrite(LED_BUILTIN, HIGH);
      delay(50);
    digitalWrite(LED_BUILTIN, LOW);
      delay(50);
  }
}
