/** 
****************************************************************************
 * @file	 HPFC_drv.h
 * @brief	Interleaved Power Factor Correction control driver Header File
 * @version  V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2023
 *****************************************************************************
 */

#ifndef _D_HPFC_DRV_H
#define _D_HPFC_DRV_H

#undef EXTERN
#ifdef DEFINE_APP
#define EXTERN
#else
#define EXTERN  extern
#endif

typedef enum {
	cPFC_OFF = 0,
	cPFC_ON
} pfc_mode_e;

typedef enum {
	cPFC_STAGE_OFF = 0,
	cPFC_STAGE_ON,
	cPFC_STAGE_EMG
} pfc_stage_e;

typedef enum {
	cPhaseA = 0,
	cPhaseB,
	cPhaseC
} phase_mode_e;

typedef enum {
	cPowerOFF = 0,
	cPowerJUDGE,
	cPowerON,
	cPowerFRQ_ERR
} power_mode_e;

typedef enum {
	cPWM_OFF = 0,
	cPWM_ON_REQ,
	cPFC_INIT2,
	cPWM_ON_SET,
	cPWM_ON
} pwm_mode_e;

typedef union {
	struct {
		uint16_t under_Vac:1;		/* reserve */
		uint16_t over_Vac:1;
		uint16_t over_Vdc:1;
		uint16_t under_Vdc:1;
		uint16_t over_iac:1;
		uint16_t ac_freq_err:1;		/* 0:normal, 1: AC Freq Err					*/
		uint16_t emg_DC:1;			/* 0:normal, 1: Over volt(soft)				*/
		uint16_t emg_S:1;			/* 0:normal, 1: Over current(soft)			*/
		uint16_t emg_H:1;			/* 0:normal, 1: Over current(hard)			*/
		uint16_t emg_H_tmp:1;		/* 0:normal, 1: Over current(hard)			*/
		uint16_t emg_comp:1;
		uint16_t comp_stop:1;
		uint16_t emg_relayon:1;
		uint16_t emg_Hz:1;
		uint16_t emg_ZC:1;
		uint16_t emg_reverse:1;
	} flg;
	uint16_t all;
} pfc_state_u;

typedef enum {
	cERR_NOT_1 = 0,			/*NOT  Err Nothing */
	cERR_VAC_OVER = 10,		/* Vac Over Voltage */
	cERR_VAC_UNDER = 11,	/*E11  Vac Under Voltage */
	cERR_VDC_OVER = 20,		/*E21  Vdc Over Voltage */
	cERR_VDC_UNDER = 21,	/*E22  Vdc Under Voltage */
	cERR_IAC_OVER = 30,		/*E30  Iac Over Current */
	cERR_I_INT = 31,		/*E31  INTE OVC Comparator detect */
	cERR_AC_OVER = 40,		/*E40  ACHZ > 66Hz */
	cERR_AC_UNDER = 41,		/*E41  ACHZ > 45Hz */
	cERR_AC_STOP = 42,		/*E42  AC Stop (Power blackout) */
} pfc_err_mode_e;

typedef struct {
	pfc_stage_e stage;						/* PFC Stage */
	pwm_mode_e pwm_mode;					/* PWM Mode */
	uint8_t control_flag;					/* control flag in pfc main loop */
	uint8_t vac_sample;						/* Vac sample cycle */
	uint8_t dn;								/* Vac high filter index */
	uint16_t max;							/* max value */
	uint16_t min;							/* min value */
	uint16_t mid;							/* middle value */
	uint32_t highfixfilter_add;				/* vac high fix value add */
	uint16_t highfixfilter[3];				/* vac data filter array */
	uint8_t over_dc_cnt;					/* Vdc over filter value */
	uint8_t over_dc_cnt_test;				/* Vdc over filter value */

	struct {
		uint16_t Err_disp_counter;			/* pfc error no use */
		uint16_t Err_mode_counter;			/* pfc error mode count */
	} Error;

	pfc_mode_e pfc_mode_req;				/* [---] */

	struct {
		uint16_t Vac_high;					/* Vac high value */
		uint16_t Vac_high_pre;				/* pre Vac high value */
		uint16_t Vac_low;					/* vac low value */
		uint16_t Vac_balance;				/* vac balance value */
		uint16_t Vac_balance_temp;			/* vac balance tmp */
		uint16_t Vac_balance_temp_pre;		/* pre vac balance tmp */
		uint32_t off_peack_time;			/* vac bottom to peak time */
		uint32_t peak_time;					/* vac peak to peak time */
		uint16_t cnt;						/* vac check basic count */
		uint8_t pos_cnt;					/* vac peak detect pos count */
		uint8_t pos_neg_cnt;				/* vac negative count */
		uint8_t judge_finish;				/* vac low high judgement finish flag */
		uint8_t judge_pos_finish;			/* vac pos judge finish flag */
		uint16_t Vac_low_fix;				/* vac low value */
		uint16_t Vac_high_fix;				/* vac high value */
		uint16_t Hz_data;					/* vac Hz check data */
		uint16_t Vdq;						/* 15% Vdq value */
		uint16_t Vdq_ave;					/* Vdq average value */
		uint32_t Vdq_sum;					/* Vdq sum */
		uint16_t cnt_vdq;					/* vdq average count */
		uint8_t Iflag;						/* vac read value 3times */
		uint16_t VacArray[3];				/* vac array for filter */
		uint16_t Vacmid;					/* vac middle value in array */
	} judgevac;

	struct {
		uint16_t	cnt;					/* vac check basic count */	
		uint16_t	balance;				/* vac balance value */
		uint16_t	low_value;				/* vac low value */
		uint32_t	bottom_cnt;				/* vac peak to bottom time */
		uint8_t		bottom_judge_filter;	/* vac negative count */
		uint8_t		bottom_judge_flag;		/* vac bottom judgement finish flag */
		uint16_t	low_fix_value;			/* vac low value */
		uint8_t		peak_judge_flag;		/* vac high judge finish flag */
		uint16_t	high_value;				/* vac high value */
		uint32_t	peak_cnt;				/* vac bottom to peak time */
		uint8_t		peak_judge_filter;		/* vac peak detect pos count */
		uint16_t	Hz_data_tmp;			/* vac Hz check data */
		uint16_t	Hz_data;				/* vac Hz check data */
		uint16_t	high_fix_value;			/* vac high value */
		uint8_t		high_fix_filter_cnt;	/* vac high filter index */
		uint16_t	high_fix_filter_max;	/* max value */
		uint16_t	high_fix_filter_min;	/* min value */
		uint16_t	high_fix_filter_mid;	/* middle value */
		uint32_t	high_fix_filter_sum;	/* vac high fix value add */
		uint16_t	high_fix_filter[3];		/* vac data filter array */

		uint8_t		low_flag;				/* vac low flag */
		uint8_t		high_flag;				/* vac high flag */
		uint8_t		zero_flag;				/* vac zero flag */
		uint8_t		zero_cnt;				/* vac zero flag */
		uint16_t	zero_protect_cnt;		/* vac zero flag */
		uint16_t	zero_protect_cnt1;		/* vac zero flag */
	}Vac; 
	
	struct {
		uint16_t update_cnt;				/* vdc update count */
		uint16_t update_cntp;				/* Vdc average count */
		uint32_t Vdc_add;					/* Vdc add */
		uint32_t Vdc_add_test;				/* no use */
		uint16_t Vac_balance;				/* vac balance */
		uint16_t Vac_balance_temp;			/* vac balance tmp */
		uint16_t cnt;						/* count */
		uint16_t Vdc_ave;					/* vdc average */
		uint16_t Vdc_ave_test;				/* no use */
		uint8_t flag;						/* vdc average flag */
		uint8_t cnt_fg;						/* vdc PI control flag */
	} dcave;

	struct {
		uint8_t update_pfc;					/* pfc pmd set flag base on status */
		uint8_t status;						/* pfc status */
		uint16_t stup_cnt;					/* pfc vdc change count at different status */
		uint8_t stop;						/* pfc stop flag */
		uint16_t decrease_cnt;				/* pfc stop status vdc decrease count */
	} pfc_signal;

	struct {
		uint16_t Vdc_ref;					/* [V/maxV] Reference value of DC voltage */
		uint16_t Vdc_lpf;					/* [V/maxV] Filter value of DC voltage */
		uint16_t Vdc_lpf0;					/* vdc lpf pre */
		uint16_t val_tgt_Vdc_temp;			/* [V/maxV] vdc target tmp value */
		uint16_t val_tgt_Vdc;				/* [V/maxV] vdc target value */
		uint16_t target_Vdc;				/* [V/maxV] vdc target value to voltage PI control */
		uint16_t target_Vdc_buf;			/* [V/maxV] vdc target buff */
		uint16_t target_Vdc_now;			/* [V/maxV] vdc target change current value */
		uint16_t target_Vdc_now_tmp;		/* [V/maxV] vdc target change current value tmp */
		uint16_t start_point_Vdc;			/* [V/maxV] vdc target value start point */
		uint16_t delta_point_Vdc;			/* [V/maxV] delta vdc */
		q15_t Iac_amp_ref;					/* [A/maxA] Reference of Iac amp */
		q15_t Iac_ref;						/* [A/maxA] Reference of Iac */
		q15_t Iac_ref_buff;					/* [A/maxA] Reference of Iac */
		q31_t Iac_amp_ref_I;				/* [A/maxA] Reference of Iac amp (Integral) */
		q31_t Iac_amp_ref_P;				/* [A/maxA] Reference of Iac amp (P) */

		uint16_t Duty_a_trg;				/* [---] A Phase Duty for pwm trigger register val */
		q31_t Duty_a_reg;					/* [---] A Phase Duty for pwm timer register val */
		q31_t Duty_a_reg0;					/* [---] A Phase Duty for pwm timer register val */

		uint16_t Iac_adc;					/* [A/MaxA]Iac adc */
		uint16_t Iac_adc0;					/* [A/MaxA]Iac adc */
		uint16_t Iac_adc1;					/* [A/MaxA]Iac adc */
		uint16_t Iac_adc2;					/* [A/MaxA]Iac adc */
		uint16_t Iac;						/* [A/MaxA]Iac adc */
		uint16_t Iac_pre;					/* [A/MaxA]Iac pre val */
		uint16_t Iac_max;					/* [A/MaxA]Iacmax val */
		uint16_t Iac_min;					/* [A/MaxA]Iac min val */
		uint16_t Iac_sum;					/* [A/MaxA]Iac sum val */
		uint16_t Iac_ave;					/* [A/MaxA]Iac average */
		uint8_t Iac_cnt;					/* Iac count */
		uint8_t Iac_cnt1;					/* Iac count1 */
		uint8_t flg;						/* no use */
		uint16_t Iac_raw;					/* [A/maxA]no filter Iac */
		uint16_t Iac_que[5];				/* no use */
		uint16_t Iao;						/* Iac zero point */

		q15_t Ila;							/* [A/maxA] a phase current */
		q15_t Ila_pre;						/* no use */
		uint16_t Vac;						/* [V/MaxV] vac val */
		uint16_t Vac_adc;					/* [V/MaxV]Vac adc */
		uint16_t Vac_adc_p;					/* [V/MaxV]no use */
		q15_t m_Vac;						/* [V/MaxV]absolute Vac */
		q15_t m_Vac_pre;					/* [V/MaxV]absolute Vac */
		q15_t Idc_avg;						/* [A/maxA] a phase current peak */
		q15_t Idc;							/* [A/maxA] a phase current peak */
		q15_t Ila_peak;						/* [A/maxA] a phase current peak */
		uint16_t Ilao_lpf;					/* [A/maxA] a phase average of zero current */

		pfc_state_u state;	/* Err state */

		struct {
			power_mode_e power_mode;		/* [---] AC power state mode */
			uint16_t interval_time;			/* [us]  AC zero-crossing interval time */
			uint16_t itime_ave;				/* [us]  AC zero-crossing interval time average */
			int32_t itime_avebuf;			/* [---] AC zero-crossing interval time average buffer */
			uint8_t ave_waitcnt;			/* [---] */
			uint16_t omega;					/* [deg/360] Omega of AC */
			uint16_t delta_omega;			/* [deg/360] Delta omega of AC */
		} ac;

		struct {
			q31_t temp_i_kp;				/* Current control proportional gain */
			q31_t temp_i_ki;				/* Current control integral gain */
		} tempcr;
	} drv;

	struct {
		uint16_t Iac_cnt;					/* no use */
		uint16_t Iac_ave;					/* no use */
		uint32_t Iac_total;					/* no use */
		uint32_t Iac_adsigma;				/* no use */
		uint16_t Iac_admax;					/* no use */
		uint16_t Iac_admin;					/* no use */
	} currentac;

	struct {
		pfc_mode_e pfc_mode, pfc_mode_pre;	/* ON/OFF mode */
		q15_t Vdc_ref;						/* [V/maxV] Reference of DC voltage */
	} usr;

	struct {
		q15_t Iac_amp_lim;				/* [A/maxA] */
		q31_t Iac_amp_lim_up;			/* [A/maxA]Iac amp limit up val */
		q31_t Iac_amp_lim_dw;			/* [A/maxA]Iac amp limit down val */
		q15_t ovc;						/* [A/maxA] */
		uint16_t Vac_max;				/* [V/MaxV]Vac max val */
		uint16_t Vdc_Limit;				/* [V/MaxV]Vdc limit val */

		struct {
			q31_t kp;					/* Voltage control proportional gain */
			q31_t ki;					/* Voltage control integral gain */
		} vol;

		struct {
			q31_t kp;					/* Current control proportional gain */
			q31_t ki;					/* Current control integral gain */
		} crt;

		struct {
			uint16_t min;				/* Duty limit */
			uint16_t max;				/* Duty limit */
		} duty_lim;

		int32_t filter;					/* vac peak/bottom val sample filter count */
		uint8_t vol_loop;				/* voltage control flag */
		q15_t pfc1;						/* trg compenstate 1 */
		q15_t pfc2;						/* trg compenstate 2 */
		uint32_u volt_up_lim;			/* voltage up limit */
		uint32_u volt_dw_lim;			/* voltage down limit */
    	uint16_t Iac_lim_up;			/* Iac lim up val */

    	uint8_t Vac_sample_time;					/*pfc vac sample time */
    	uint16_t pwm_prd_time;						/*pwm period time */
    	uint16_t trg_switch_point;					/*pfc trigger switch point */
    	uint16_t stup_cnt;							/*pfc stop count */
    	uint8_t Vdc_target_cnt; 					/*vdc target count */
    	uint8_t Vadd;								/*delta vdc valve val */
    	uint8_t Vdc_tgt_cycle;						/*vdc control cycle */
    	uint8_t Vdc_dw_limt;						/*vdc down limit */
	} para;

	struct {
		TSB_T32A_TypeDef *T32AxPWMA;		/* MCU unit address of Timer for PWM A */
		TSB_T32A_TypeDef *T32AxTTRG;		/* MCU unit address of Timer for AD Triger */
		TSB_T32A_TypeDef *T32AxZEROX;		/* MCU unit address of Timer for ZeroCross */
		TSB_AD_TypeDef *ADx;				/* MCU unit address of ADC for PFC */
	} mcu_unit;
} pfc_t;

/*****************************************************************
 * Definition of variable
 *****************************************************************/
#define cVAC_CYCLE						(10000) 								/* [us]Vac cycle 10ms */
#define cZERO							(5000)									/* [us] zero later 0.5ms*/
#define cPREZERO_CNT					((cVAC_CYCLE-cZERO)/cPWMPRD_PFC)	    /*Pre zero point*/

#define cVOLT_UP_LIM_PFC				(20.0f)									/* [V/s] Rate of volatge rise */
#define cVOLT_DW_LIM_PFC				(20.0f)									/* [V/s] Rate of voltage drop */
#define cVdc_Cycle						(250 * 1e-6)							/* [s] Vdc control cycle*/																	

EXTERN pfc_t Hpfc;
EXTERN uint8_t	g_CntTgtVdc1;
EXTERN uint8_t	g_CntTgtVdc2;
EXTERN uint8_t	g_CntTgtVdc3;
EXTERN int16_t	g_SubTgtVdc;

/*===================================================================*
	  Proto Type Definition
 *===================================================================*/
EXTERN void init_HPFC_Control(void);
EXTERN void HPFC_Control_Int(void);
EXTERN bool HPFC_SetPFC_Mode(pfc_mode_e _mode);
EXTERN void HPFC_INT_ADC_Fin(void);
EXTERN void HPFC_INT_ERROR_CHECK(void);
EXTERN void HPFC_PWMOutputOn_pre(pfc_t * const _pfc);
EXTERN void HPFC_PWMOutputOn(pfc_t * const _pfc);
EXTERN void HPFC_PWMOutputOff(pfc_t * const _pfc);
EXTERN void HPFC_GetPhaseCurrentADC(pfc_t * const _pfc);
EXTERN void HPFC_CalPhaseCurrent(pfc_t * const _pfc);
EXTERN void HPFC_CalPWM(pfc_t * const _pfc);
EXTERN void HPFC_CalZeroCurrent(pfc_t * const _pfc);
EXTERN void HPFC_CalVdcFilter(pfc_t * const _pfc);
EXTERN void HPFC_UserControl(void);
EXTERN void HPFC_Control_Main(void);
EXTERN void HPFC_CalVdcFilter_16k(pfc_t * const _pfc);
EXTERN void HPFC_INT_ERROR_CHECK_16k(void);
EXTERN void HPFC_VacVdc_Handle(pfc_t* const _pfc);
EXTERN void HPFC_CalVacFilter(pfc_t * const _pfc);
EXTERN void HPFC_ControlCurrent(pfc_t * const _pfc, phase_mode_e _mode);
EXTERN void HPFC_ControlVoltage(pfc_t * const _pfc);
EXTERN void HPFC_INT_pwmA(void);
EXTERN void HPFC_VdcTarget(pfc_t * const _pfc, vector_t* const _motor);
EXTERN void HPFC_CalVacFilter_16k(pfc_t * const _pfc);
EXTERN void HPFC_ControlVoltage_16k(pfc_t * const _pfc);
#undef	EXTERN
#endif				/* _D_HPFC_DRV_H */

/*************************** END OF FILE **************************************/
