/**
 ****************************************************************************
 * @file    ipdrv_enc.h
 * @brief   This file provides all the functions prototypes for M4Kx A-ENC driver. 
 * @version V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2023
 *****************************************************************************
 */
#if (defined(__A_ENC) ||defined(__ENC))
/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __IPDRV_ENC_H
#define __IPDRV_ENC_H

#ifdef __cplusplus
extern "C" {
#endif

/* Includes ------------------------------------------------------------------*/
#include "TMPM4KyA.h"
#include "ipdrv_common.h"

/** @addtogroup TX04_Periph_Driver
  * @{
  */
/** @addtogroup ENC
  * @{
  */

/** @addtogroup ENC Exported_types
  * @{
  */

/** @brief ENC input control structure definition
  * @{
  */

    typedef struct {
        uint32_t ModeType;      /* Encoder input mode selection */
        uint32_t PhaseType;     /* 2-phase / 3-phase input selection (sensor mode) */
        uint32_t EdgeType;      /* Edge selection of Z-phase (timer mode/phase counter mode) */
        uint32_t ZphaseStatus;  /* Enable or disable the Z-phase (Encoder mode/timer mode/phase counter mode) */
        uint32_t SetClearCount; /* Select the clear a match for Counter (timer mode) */
        uint32_t UpDownType;    /* Set the Up/Down counter mode (sensor phase mode/phase count mode) */
        uint32_t CountOperation;        /* Set Reload Register of count Enable/Disable */
        uint32_t CompareMode;   /* Compare  match or size (timer mode/sensor timer mode) */
        uint32_t DetectionDirection;    /* Set the detection direction for the decoder (encoder mode/sensor mode) */
        uint32_t SkipDetection; /* Sets skip detection (sensor mode) */
        uint32_t SelectCapture; /* Select trigger capture operation (sensor timer mode/sensor phase mode/timer mode/phase counter mode) */
        uint32_t PulseDivFactor;        /* Set the encoder pulse division factor (encoder mode/sensor event mode) */
    } ENC_InitTypeDef;

/** @brief ENC status flag union definition
  * @{
  */
    typedef union {
        uint32_t All;
        struct {
            uint32_t InputERRFlag:1;    /* bit 0, abnormal error (sensor mode) */
            uint32_t DecodeERRFlag:1;   /* bit 1, decoding detection error (encoder mode/sensor mode) */
            uint32_t SkipDetectionFlag:1;       /* bit 2, skip detection flag */
            uint32_t Reserved1:9;       /* bit 3 to bit 11, Reserved */
            uint32_t ZPhaseDetectFlag:1;        /* bit 12,ENC Z-phase detect flag */
            uint32_t RotationDirection:1;       /* bit 13,ENC rotation direction (encoder mode/sensor mode) */
            uint32_t ReverseErrorFlag:1;        /* bit 14,ENC sensor mode (at time count or phase count) reverse error flag */
            uint32_t Reserved2:17;      /* bit 15 to bit 31, Reserved */
        } Bit;
    } ENC_FlagStatus;

/* End of ENC_Exported_types */

/** @defgroup ENC_Exported_Constants
  * @{
  */
#define EN0    TSB_EN0
#define EN1    TSB_EN1
#define EN2    TSB_EN2
#define IS_ENC_PERIPH(param)          ((param) == EN0 || (param) == EN1 || (param) == EN2)

#define ENC_ENCODER_MODE              ((uint32_t)0x00000000)
#define ENC_SENSOR_EVENT_MODE         ((uint32_t)0x00000001)
#define ENC_SENSOR_TIME_MODE          ((uint32_t)0x00000002)
#define ENC_TIMER_MODE                ((uint32_t)0x00000003)
#define ENC_SENSOR_PHASE_MODE         ((uint32_t)0x00000006)
#define ENC_PHASE_COUNTER_MODE        ((uint32_t)0x00000007)
#define IS_ENC_MODE(param)            (((param) == ENC_ENCODER_MODE) || \
                                       ((param) == ENC_SENSOR_EVENT_MODE) || \
                                       ((param) == ENC_SENSOR_TIME_MODE) || \
                                       ((param) == ENC_TIMER_MODE) || \
                                       ((param) == ENC_SENSOR_PHASE_MODE) || \
                                       ((param) == ENC_PHASE_COUNTER_MODE))


#define ENC_TWO_PHASE                 ((uint32_t)0x00000000)
#define ENC_THREE_PHASE               ((uint32_t)0x00000001)
#define IS_ENC_PHASE(param)           (((param) == ENC_TWO_PHASE) || \
                                       ((param) == ENC_THREE_PHASE))

#define ENC_ZPHASE_DISABLE            ((uint32_t)0x00000000)
#define ENC_ZPHASE_ENABLE             ((uint32_t)0x00000001)
#define IS_ENC_ZPHASE_STATUS(param)   (((param) == ENC_ZPHASE_ENABLE) || \
                                       ((param) == ENC_ZPHASE_DISABLE))

#define ENC_RISING_EDGE               ((uint32_t)0x00000001)
#define ENC_FALLING_EDGE              ((uint32_t)0x00000002)
#define ENC_BOTHEDGES_EDGE            ((uint32_t)0x00000003)
#define IS_ENC_EDGE(param)            (((param) == ENC_RISING_EDGE) || \
                                       ((param) == ENC_FALLING_EDGE) || \
                                       ((param) == ENC_BOTHEDGES_EDGE))

#define ENC_CLEAR_INT                 ((uint32_t)0x00000000)
#define ENC_CLEAR_RELOAD              ((uint32_t)0x00000001)
#define IS_ENC_CLEAR_CONDITION(param) (((param) == ENC_CLEAR_INT) || \
                                       ((param) == ENC_CLEAR_RELOAD))

#define ENC_COUNT_UP                  ((uint32_t)0x00000000)
#define ENC_COUNT_DOWN                ((uint32_t)0x00000001)
#define ENC_COUNT_AUTO                ((uint32_t)0x00000002)
#define IS_ENC_COUNT_MODE(param)      (((param) == ENC_COUNT_UP) || \
                                       ((param) == ENC_COUNT_DOWN) || \
                                       ((param) == ENC_COUNT_AUTO))

#define ENC_COUNT_CONTINUES           ((uint32_t)0x00000000)
#define ENC_COUNT_STOP                ((uint32_t)0x00000001)
#define IS_ENC_COUNT_STOP(param)      (((param) == ENC_COUNT_CONTINUES) || \
                                       ((param) == ENC_COUNT_STOP))

#define ENC_COMPARE_MATCH             ((uint32_t)0x00000000)
#define ENC_COMPARE_SIZE              ((uint32_t)0x00000001)
#define IS_ENC_COMPARE_MODE(param)    (((param) == ENC_COMPARE_MATCH) || \
                                       ((param) == ENC_COMPARE_SIZE))

#define ENC_CW_CCW_EDGE               ((uint32_t)0x00000000)
#define ENC_CW_EDGE                   ((uint32_t)0x00000001)
#define ENC_CCW_EDGE                  ((uint32_t)0x00000002)
#define ENC_CW_CCW_MAINTAIN           ((uint32_t)0x00000003)
#define IS_ENC_EDGE_DIRECTION(param)  (((param) == ENC_CW_CCW_EDGE) || \
                                       ((param) == ENC_CW_EDGE) || \
                                       ((param) == ENC_CCW_EDGE) || \
                                       ((param) == ENC_CW_CCW_MAINTAIN))

#define ENC_DETECTION_DISABLE         ((uint32_t)0x00000000)
#define ENC_DETECTION_ENABLE          ((uint32_t)0x00000001)
#define IS_ENC_DETECT_SKIP(param)     (((param) == ENC_DETECTION_ENABLE) || \
                                       ((param) == ENC_DETECTION_DISABLE))

#define ENC_CAPTURE_CLEAR             ((uint32_t)0x00000000)
#define ENC_CAPTURE_ONLY              ((uint32_t)0x00000001)
#define IS_ENC_CAPTURE_MODE(param)    (((param) == ENC_CAPTURE_CLEAR) || \
                                       ((param) == ENC_CAPTURE_ONLY))

/* Encoder pulse division factor */
#define ENC_PULSE_DIV1                ((uint32_t)0x00000000)
#define ENC_PULSE_DIV2                ((uint32_t)0x00000001)
#define ENC_PULSE_DIV4                ((uint32_t)0x00000002)
#define ENC_PULSE_DIV8                ((uint32_t)0x00000003)
#define ENC_PULSE_DIV16               ((uint32_t)0x00000004)
#define ENC_PULSE_DIV32               ((uint32_t)0x00000005)
#define ENC_PULSE_DIV64               ((uint32_t)0x00000006)
#define ENC_PULSE_DIV128              ((uint32_t)0x00000007)
#define IS_ENC_DIV_FACTOR(param)      (((param) == ENC_PULSE_DIV1) || \
                                       ((param) == ENC_PULSE_DIV2) || \
                                       ((param) == ENC_PULSE_DIV4) || \
                                       ((param) == ENC_PULSE_DIV8) || \
                                       ((param) == ENC_PULSE_DIV16) || \
                                       ((param) == ENC_PULSE_DIV32) || \
                                       ((param) == ENC_PULSE_DIV64) || \
                                       ((param) == ENC_PULSE_DIV128))

#define ENC_SAMPLE_CLK_FSYS_1         ((uint32_t)0x00000000)
#define ENC_SAMPLE_CLK_FSYS_2         ((uint32_t)0x00000001)
#define ENC_SAMPLE_CLK_FSYS_4         ((uint32_t)0x00000002)
#define ENC_SAMPLE_CLK_FSYS_8         ((uint32_t)0x00000003)
#define IS_ENC_FSYS_FACTOR(param)     (((param) == ENC_SAMPLE_CLK_FSYS_1) || \
                                       ((param) == ENC_SAMPLE_CLK_FSYS_2) || \
                                       ((param) == ENC_SAMPLE_CLK_FSYS_4) || \
                                       ((param) == ENC_SAMPLE_CLK_FSYS_8))

#define ENC_PWM_SAMPLE_DISABLE        ((uint32_t)0x00000000)
#define ENC_PWM_SAMPLE_ENABLE         ((uint32_t)0x00000001)
#define IS_ENC_PWM_SAMPLEMD(param)    (((param) == ENC_PWM_SAMPLE_DISABLE) || \
                                       ((param) == ENC_PWM_SAMPLE_ENABLE))

#define ENC_PWM_ON_PERIOD             ((uint32_t)0x00000000)
#define ENC_PWM_OFF_EDGE              ((uint32_t)0x00000001)
#define IS_ENC_EDGE_SELECT(param)     (((param) == ENC_PWM_ON_PERIOD) || \
                                       ((param) == ENC_PWM_OFF_EDGE))

#define ENC_PWM_COUNT_STOP            ((uint32_t)0x00000000)
#define ENC_PWM_COUNT_STOP_CLEAR      ((uint32_t)0x00000001)
#define IS_ENC_PWM_COUNTMD(param)     (((param) == ENC_PWM_COUNT_STOP) || \
                                       ((param) == ENC_PWM_COUNT_STOP_CLEAR))

#define IS_ENC_16BITVALUE(param)      (((param) <= 0xFFFFU))
#define IS_ENC_8BITVALUE(param)       (((param) <= 0xFFU))
#define IS_ENC_7BITVALUE(param)       (((param) <= 0x7FU))

/** @} */
/* End of group ENC_Exported_Constants */


/** @defgroup ENC_Exported_FunctionPrototypes
  * @{
  */
    void ENC_Enable(TSB_EN_TypeDef * ENx);
    void ENC_Init(TSB_EN_TypeDef * ENx, ENC_InitTypeDef * InitStruct);
    uint32_t ENC_GetControlMode(TSB_EN_TypeDef * ENx);
    ENC_FlagStatus ENC_GetENCFlag(TSB_EN_TypeDef * ENx);
    void ENC_SetCounterReload(TSB_EN_TypeDef * ENx, uint32_t PeriodValue);
    uint32_t ENC_GetCounterValue(TSB_EN_TypeDef * ENx);
    void ENC_SetNoiseCancelTime(TSB_EN_TypeDef * ENx, uint32_t Time);
    void ENC_SetSampleClk(TSB_EN_TypeDef * ENx, uint32_t SampleClk);

/** @} */
/* End of group ENC_Exported_FunctionPrototypes */

/** @} */
/* End of group ENC */

/** @} */
/* End of group TX04_Periph_Driver */

#ifdef __cplusplus
}
#endif                          /* __cplusplus */
#endif                          /* __IPDRV_ENC_H */
#endif
