/**
 ****************************************************************************
 * @file     usercon.c
 * @brief    User Control
 * @version  V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENCE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2021
 *****************************************************************************
 */
#include <stdlib.h>
#include <stdio.h>

#define DEFINE_APP
#include "usercon.h"
#undef	DEFINE_APP
#include "ipdefine.h"

/*===================================================================*
	  Macro Definition
 *===================================================================*/

/*===================================================================*
	  Local Constant Data Definition
 *===================================================================*/
/* Timer setting */
#define c100MS_TIMER		(100)							/* [ms] (4kHz * 4) * 100 */
#define c250MS_TIMER		(250)							/* [ms] (4kHz * 4) * 250  */
#define c1000MS_TIMER		(1000)							/* [ms] (4kHz * 4) * 1000 */
#define c3000MS_TIMER		(3000)							/* [ms] (4kHz * 4) * 3000 */
#define cPORT_WAIT_CNT		(1000)							/* Wait count for initialization of UART port */


/* Key settting */
#define P_PSW1				TSB_PD_DATA_PD0 				/* PushSW1 */
#define P_PSW2				TSB_PD_DATA_PD1 				/* PushSW2 */
#define P_PSW3				TSB_PE_DATA_PE3 				/* PushSW3 */
#define P_PSW4				TSB_PE_DATA_PE4 				/* PushSW4 */
#define P_PSW5				TSB_PE_DATA_PE5 				/* PushSW5 */
#define P_PSW6				TSB_PU_DATA_PU3 				/* PushSW6 */

#define cKEY_CHATA_CNT		(20)							/* [cnt] chattering counter for KEY SW */

#define cKEY_NO_INPUT		(0x00)							/* PushSW1`6 no input */
#define cKEY_MASK_PSW		(0xFC)							/* PushSW1`6 */
#define cKEY_INPUT_PSW1		(0x80)							/* PushSW1 */
#define cKEY_INPUT_PSW2		(0x40)							/* PushSW2 */
#define cKEY_INPUT_PSW3		(0x20)							/* PushSW3 */
#define cKEY_INPUT_PSW4		(0x10)							/* PushSW4 */
#define cKEY_INPUT_PSW5		(0x08)							/* PushSW5 */
#define cKEY_INPUT_PSW6		(0x04)							/* PushSW6 */
#define LED_UART_ERR		TSB_PK_DATA_PK1 				/* LED1 */
#define LED_PFC_ON			TSB_PK_DATA_PK2 				/* LED2 */
#define LED_MOTOR_CH0		TSB_PK_DATA_PK3 				/* LED3 */
#define LED_CH0_EMG			TSB_PJ_DATA_PJ5 				/* LED4 */
#define LED_CH1_EMG			TSB_PV_DATA_PV1 				/* LED5 */
#define LED_CH2_EMG			TSB_PV_DATA_PV0 				/* LED6 */

#define	cLED_ON				(1)								/* LED ON level */
#define	cLED_OFF			(0)								/* LED OFF level */


/* Speed control setting */

#define	cMOTOR_SPEED_NOCHG			(0x00)					/* Target speed of motor not change */
#define	cMOTOR_SPEED_UP				(0x01)					/* Target speed of motor +10Hz */
#define	cMOTOR_SPEED_DOWN			(0x02)					/* Target speed of motor -10Hz */
#define	cMOTOR_SPEED_STOP			(0x03)					/* Motor stop */

#define	cSPEED_USER_UP_DOWN			(10)					/* [Hz] Target speed change of motor */
#define	cSPEED_USER_MIN				(60)					/* [Hz] Min Target speed of motor */
#define	cSPEED_USER_MAX				(120)					/* [Hz] Max Target speed of motor */

#define	cCONTROL_MOTOR_CH0			(0x00)					/* Target motor channel0 */
#define	cCONTROL_MOTOR_CH1			(0x01)					/* Target motor channel1 */

/* State control setting */
#define	cPFC_OFF					(0)						/* PFC OFF */
#define	cPFC_ON						(1)						/* PFC ON */

#define	cDAC_MDOE0					(0x00)					/* Display dac mode 0 */
#define	cDAC_MDOE1					(0x01)					/* Display dac mode 1 */
#define	cDAC_MDOE2					(0x02)					/* Display dac mode 2 */
#define	cDAC_MDOE3					(0x03)					/* Display dac mode 3 */

#define	cDISP_CH0					(0x00)					/* Display channel 0 */
#define	cDISP_CH1					(0x01)					/* Display channel 1 */
#define	cDISP_CH2					(0x02)					/* Display channel 2 */


/* UART Setting */
#define cRESULT_NG					(0)						/* deta check err */
#define cRESULT_OK					(1)						/* deta check ok */

#define cSEND_DATA_NUM				(6)						/* Send data size */
#define cRECEIVE_DATA_NUM			(7)						/* Receive data size */

#define cUART_START_COMMAND			(0x00)					/* UART mode : start command */
#define cUART_SEND_WAIT				(0x01)					/* UART mode : data send wait */
#define cUART_RECEIVE_WAIT			(0x02)					/* UART mode : data receive wait */
#define cUART_ERR					(0x03)					/* UART mode : error */

#define cSEND_IDLE					(0x00)					/* Send mode : idle */
#define cSEND_START_COMMAND			(0x01)					/* Send mode : send start command */
#define cSEND_STATUS_DISP			(0x02)					/* Send mode : motor status display request */
#define cSEND_ROTATE_DISP			(0x03)					/* Send mode : motor speed display request */
#define cSEND_ROTATE_SPEED			(0x04)					/* Send mode : send target speed */
#define cSEND_CHANGE_DISP			(0x05)					/* Send mode : Display change request */

#define cDISP_IDLE					(0x00)					/* Disp mode : idle */
#define cDISP_STATUS				(0x01)					/* Disp mode : motor status */
#define cDISP_ROTATE_SPEED			(0x02)					/* Disp mode : motor speed */

#define cREQ_SYSTEM_START			(0x10)					/* System start request */
#define cREQ_ROTATE_MOTOR			(0x11)					/* Target speed update request*/
#define cREQ_CANGE_MOTOR			(0x12)					/* Target motor change request*/
#define cREQ_ON_OFF_PFC				(0x13)					/* PFC state change request*/
#define cREQ_ALL_STOP_MOTOR			(0x14)					/* All motor stop request*/

#define cREQ_STATUS_CH				(0x15)					/* Displayed channel change request*/
#define cREQ_STATUS_DAC				(0x16)					/* Displayed DACmode change request*/
#define cGET_MOTOR_ENABLE			(0x80)					/* Operating status */
#define cGET_STATE_EMG				(0x81)					/* Emergency status */
#define cGET_STAGE					(0x82)					/* Main stage */
#define cGET_CONTROL_CH				(0x83)					/* Control channel */
#define cGET_CARRIER_FREQUENCY		(0x84)					/* Carrier frequency */
#define cGET_MOTOR_SPEED_MIN		(0x85)					/* Minimum rotation speed */
#define cGET_MOTOR_SPEED_MAX		(0x86)					/* Maximum rotation speed */
#define cGET_DEAD_TIME				(0x87)					/* Dead time */
#define cGET_GATE_ACTIVE			(0x88)					/* Gate active */
#define cGET_POSITION_DITECT		(0x89)					/* Position detect */
#define cGET_VDC_VOLTAGE			(0x8A)					/* VDC voltage */
#define cGET_OUTROOM_TEMP			(0x8B)					/* Outroom temp */
#define cGET_OUTPIPE_TEMP			(0x95)					/* Outpipe temp */
#define cGET_EXHAUST_TEMP			(0x96)					/* Exhaust temp */
#define cGET_DIODE_TEMP				(0x97)					/* Diode temp */
#define cGET_IGBT_TEMP				(0x98)					/* IGBT temp */
#define cGET_HVMOS_TEMP				(0x99)					/* HVMOS temp */
#define cGET_U_VOLTAGE				(0x8C)					/* U-phase voltage */
#define cGET_V_VOLTAGE				(0x8D)					/* V-phase voltage */
#define cGET_W_VOLTAGE				(0x8E)					/* W-phase voltage */
#define cGET_DAC_DATA				(0x8F)					/* Dac data */
#define cGET_INTERNAL_AMP			(0x90)					/* Internal amp */
#define cGET_DIRECTION				(0x91)					/* Direction */
#define cGET_MODULATION				(0x92)					/* Modulation */
#define cGET_KEY_OPERATION			(0x93)					/* Key operation */
#define cGET_MOTOR_SPEED			(0x94)					/* motor rotation speed */

#define cEMGBIT_MASK				(0x04)					/* Emergency bit */
#define cNABIT_MASK					(0x02)					/* Not abailable bit */
#define cACKBIT_MASK				(0x01)					/* Ack bit */

#define cDISP_CONTROL_CH			(0x00)					/* Control channel display */
#define cDISP_CARRIER_FREQUENCY		(0x01)					/* Carrier frequency display */
#define cDISP_MOTOR_SPEED_MIN_MAX	(0x02)					/* Minimam/Maximum rotation speed display */
#define cDISP_DEAD_TIME				(0x03)					/* Dead time display */
#define cDISP_GATE_ACTIVE			(0x04)					/* Gate active display */
#define cDISP_POSITION_DITECT		(0x05)					/* position detect display */
#define cDISP_VDC_VOLTAGE			(0x06)					/* VDC voltage display */
#define cDISP_OUTROOM_TEMP			(0x07)					/* Outroom temp display */
#define cDISP_OUTPIPE_TEMP			(0x08)					/* Outpipe temp display */
#define cDISP_EXHAUST_TEMP			(0x09)					/* Exhaust temp display */
#define cDISP_DIODE_TEMP			(0x0A)					/* Diode temp display */
#define cDISP_IGBT_TEMP				(0x0B)					/* IGBT temp display */
#define cDISP_HVMOS_TEMP			(0x0C)					/* HVMOS temp display */
#define cDISP_UVW_VOLTAGE			(0x0D)					/* U-phase/V-phase/W-phase voltage display */
#define cDISP_DAC_DATA				(0x0E)					/* Dac data display */
#define cDISP_INTERNAL_AMP			(0x0F)					/* Internal amp display */
#define cDISP_DIRECTION				(0x10)					/* Direction display */
#define cDISP_MODULATION			(0x11)					/* Modulation display */

/* User Setting */
#define cMOTOR_DISABLE				(0)						/* motor operation : disable */
#define cMOTOR_ENABLE				(1)						/* motor operation : enable */

#define	cGATE_ACTIVE_H_H			(0)						/* Gate active : H/H */
#define	cGATE_ACTIVE_L_L			(1)						/* Gate active : L/L */
#define	cGATE_ACTIVE_H_L			(2)						/* Gate active : H/L */
#define	cGATE_ACTIVE_L_H			(3)						/* Gate active : L/H */

#define	cPOSITION_DETECT_3SHUNT		(0)						/* Position Ditect : 3shunt */
#define	cPOSITION_DETECT_1SHUNT		(1)						/* Position Ditect : 1shunt */

#define	cINTERNAL_AMP_NO			(0)						/* Internal amp : External */
#define	cINTERNAL_AMP_YES			(1)						/* Internal amp : Internal */

#define cDIRECTION_CW				(0)						/* Direction : plus */
#define cDIRECTION_CCW				(1)						/* Direction : minus */

#define	cMODULATION_3PHASE			(0)						/* Modulation : 3phase */
#define	cMODULATION_2PHASE			(1)						/* Modulation : 2phase */

#define cSPDCTL_VR					(0)						/* Speed control : volume */
#define cSPDCTL_PUSH				(1)						/* Speed control : push sw */

#define cMOTOR_CH0					(0x00)					/* Control channel : CH0 */
#define cMOTOR_CH1					(0x01)					/* Control channel : CH1 */
#define cMOTOR_CH2					(0x02)					/* Control channel : CH2 */

/*===================================================================*
	  Local Variable Definition
 *===================================================================*/
static sw_t			keydata_old;							/* Previous Key data */
static sw_t			portbuf; 								/* Key port data */
static sw_t			portbuf_old; 							/* Previous Key port data */
static uint8_t		key_chatacnt;							/* Chattering counter for Key */

static uint8_t		motor_action;							/* Rotation speed flucuation */
static uint8_t		uarterr_timer;							/* UART err timer(250ms blink) */
static uint16_t		rotatedisp_timer;						/* Rotation speed display timer(1000ms) */
static uint8_t		timeout_timer;							/* 100ms timer counter */
static uint8_t		emg_timer;								/* EMG timer(250ms blink) */
static uint16_t		keylongpush_timer;						/* KeyLongPush timer(3000ms) */


static int32_t		speed_temp;								/* Motor target speed */
static int32_t		speed_temp_ch0;							/* Motor target speed (CH0) */
static int32_t		speed_temp_ch1;							/* Motor target speed (CH1) */

static uint8_t		ctrl_motor;								/* Target Motor (CH0 or CH1)*/
static uint8_t		pfc_state;								/* PFC State */
static uint8_t		dac_mode;								/* DAC mode */
static uint8_t		disp_ch;								/* Display CH */


static uint8_t		led1_state;								/* LED1 lighting state */
static uint8_t		led4_state;								/* LED4 lighting state */
static uint8_t		led5_state;								/* LED5 lighting state */
static uint8_t		led6_state;								/* LED6 lighting state */

static uint8_t		uart_mode;								/* uart control mode */
static uint8_t		send_mode;								/* uart send mode */
static uint8_t		sendbuffer[cSEND_DATA_NUM];				/* Send data buffer */
static uint8_t		receivebuffer[cRECEIVE_DATA_NUM];		/* Receive data buffer */
static uint8_t		receivedata[cRECEIVE_DATA_NUM];			/* Receive data */
static uint8_t		txpnt;									/* Pointer of send data  */
static uint8_t		rxpnt;									/* Pointer of receive data  */

static uint8_t		disp_mode;								/* terminal display mode */
static uint8_t		command_id;								/* data command id */
static uint8_t		receive_checksum;						/* receive data checksum */
static uint8_t		uart_err_count;							/* uart receive err counter */

static uint8_t		disp_id;								/* terminal display command id */

static uint8_t		state_emg[2];							/* motor emg state */
static uint8_t		main_stage[2];							/* motor main stage */
static uint8_t		control_ch[2];							/* motor control ch */
static uint32_t		carrier_frequency[2];					/* [MHz] motor carrier frequency */
static uint32_t		motor_speed_min[2];						/* [Hz] motor min speed */
static uint32_t		motor_speed_max[2];						/* [Hz] motor max speed */
static float32_t	dead_time[2];							/* [us] motor dead time */
static uint8_t		gate_active[2];							/* motor gate active */
static uint8_t		position_detect[2];						/* motor position detect*/
static float32_t	vdc_voltage[2];							/* [V] motor vdc voltage */
static float32_t	outroom_temp[3];						/* outroom temp */
static float32_t	outpipe_temp[3];						/* outpipe temp */
static float32_t	exhaust_temp[3];						/* exhaust temp */
static float32_t	diode_temp[3];							/* diode temp */
static float32_t	igbt_temp[3];							/* igbt temp */
static float32_t	hvmos_temp[3];							/* hvmos temp */
static float32_t	u_voltage[2];							/* [V] motor u voltage */
static float32_t	v_voltage[2];							/* [V] motor v voltage */
static float32_t	w_voltage[2];							/* [V] motor w voltage */
static uint8_t		dac_data[8];							/* motor dac data */
static uint8_t		internal_amp[2];						/* motor internal amp */
static uint8_t		direction[2];							/* motor direction */
static uint8_t		modulation[2];							/* motor modulation */
static int8_t		motor_speed[2];							/* [Hz] motor speed */

/*===================================================================*
	  Local Proto Type Definition
 *===================================================================*/
/***** Switch control *****/
static void Keychata(sw_t portdata);


/***** UART control *****/
static void send_data_set(uint8_t id);
static uint8_t data_check(void);
static void uart_send_start(void);
static void uart_send_stop(void);
static void terminaldisp_dataset(void);


/**
 *********************************************************************************************
 * @brief		timer count.
 *
 * @return		none
 *********************************************************************************************
 */
void timer_count(void)
{
	if(uart_mode == cUART_ERR)
	{
		if(uarterr_timer < c250MS_TIMER)
		{
			if(++uarterr_timer >= c250MS_TIMER)
			{
				uarterr_timer = 0;
				led1_state ^= 1;
			}
		}
	}
	else
	{
		if(rotatedisp_timer < c1000MS_TIMER)
		{
			if(++rotatedisp_timer >= c1000MS_TIMER)
			{
				rotatedisp_timer = 0;
				flg.disp.get_status_2 = 1;
			}
		}
	}

	if(uart_mode != cUART_ERR)
	{
		if(flg.uart.emg_bit == 1)
		{
			if(emg_timer < c250MS_TIMER)
			{
				if(++emg_timer >= c250MS_TIMER)
				{
					emg_timer = 0;
					switch(control_ch[1])
					{
						case cMOTOR_CH0:
							led4_state ^= 1;
							break;
					
						case cMOTOR_CH1:
							led5_state ^= 1;
							break;
					
						case cMOTOR_CH2:
							led6_state ^= 1;
							break;
					
						default:
							break;
					}
				}
			}
		}
		else
		{
			if(timeout_timer < c100MS_TIMER)
			{
				if(++timeout_timer >= c100MS_TIMER)
				{
					timeout_timer = 0;
					rxpnt = 0;
					flg.timer.timeout = 1;
				}
			}
		}
		if(keylongpush_timer > 0)
		{
			if(--keylongpush_timer <= 0)
			{
				if((keydata.all & cKEY_MASK_PSW) == cKEY_INPUT_PSW6)
				{
					flg.timer.keylongpush = 1;
				}
			}
		}
	}

}


/**
 *********************************************************************************************
 * @brief		Initialize key.
 *
 * @return		none
 *********************************************************************************************
 */
void init_Uikey(void)
{
	/* Supply clock fsys to PORT */
	CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_PORTD, ENABLE);	/* PushSW1-2 */
	CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_PORTE, ENABLE);	/* PushSW3-5 */
	CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_PORTU, ENABLE);	/* PushSW6 */
	
	/* Select input mode */
	TSB_PD_IE_PD0IE = ENABLE;							// Input enable for PD0
	TSB_PD_IE_PD1IE = ENABLE;							// Input enable for PD1
	TSB_PE_IE_PE3IE = ENABLE;							// Input enable for PE3
	TSB_PE_IE_PE4IE = ENABLE;							// Input enable for PE4
	TSB_PE_IE_PE5IE = ENABLE;							// Input enable for PE5
	TSB_PU_IE_PU3IE = ENABLE;							// Input enable for PU3
}

/**
 *********************************************************************************************
 * @brief		keyscan
 *
 * @return		none
 *********************************************************************************************
 */
void Uikeyscan(void)
{
	portbuf.bit.sw1 = P_PSW1;
	portbuf.bit.sw2 = P_PSW2;
	portbuf.bit.sw3 = P_PSW3;
	portbuf.bit.sw4 = P_PSW4;
	portbuf.bit.sw5 = P_PSW5;
	portbuf.bit.sw6 = P_PSW6;
	Keychata(portbuf);

	if (flg.key.keydata_set == 1)
	{
		flg.key.keydata_set = 0;

		if ((keydata_old.all & cKEY_MASK_PSW) == cKEY_NO_INPUT)
		{
			keylongpush_timer = 0;
			flg.key.keylongpush_done = 0;
			switch (keydata.all & cKEY_MASK_PSW)
			{

				case cKEY_INPUT_PSW1:
					motor_action = cMOTOR_SPEED_STOP;
					break;

				case cKEY_INPUT_PSW2:
					flg.motor.motor_pfcchange = 1;
					break;

				case cKEY_INPUT_PSW3:
					flg.motor.motor_change = 1;
					break;

				case cKEY_INPUT_PSW4:
					motor_action = cMOTOR_SPEED_UP;
					break;

				case cKEY_INPUT_PSW5:
					motor_action = cMOTOR_SPEED_DOWN;
					break;

				case cKEY_INPUT_PSW6:
					keylongpush_timer = c3000MS_TIMER;
					break;
                
				default:
					break;
			}
		}

		else
		{
			if(flg.key.keylongpush_done != 1)
			{
				if((keydata_old.all & cKEY_MASK_PSW) == cKEY_INPUT_PSW6)
				{
					if((keydata.all & cKEY_MASK_PSW) == cKEY_NO_INPUT)
					{
						keylongpush_timer = 0;
						flg.disp.change_dac = 1;
					}
				}
				else
				{
					keylongpush_timer = 0;
				}
			}
		}
		keydata_old = keydata;
	}
	else
	{
		if(flg.key.keylongpush_done != 1)
		{
			if(flg.timer.keylongpush == 1)
			{
				flg.timer.keylongpush = 0;
				flg.disp.change_ch = 1;
			}
		}
	}
}


/**
 *********************************************************************************************
 * @brief		keyscan
 *
 * @return		none
 *********************************************************************************************
 */
void UikeyInitscan(void)
{
	key_chatacnt = cKEY_CHATA_CNT;
	
	while(flg.key.keydata_set == 0)
	{
		if (M_Main_Counter >= cMAINLOOP_CNT)
		{
			M_Main_Counter = 0;

			/***** Clear WDT *****/
			SIWDT_WriteClearCode();

			*((uint8_t*)&portbuf) = 0xfc;							/* PushSW low active */
			portbuf.bit.sw1 = P_PSW1;
			portbuf.bit.sw2 = P_PSW2;
			portbuf.bit.sw3 = P_PSW3;
			portbuf.bit.sw4 = P_PSW4;
			portbuf.bit.sw5 = P_PSW5;
			portbuf.bit.sw6 = P_PSW6;

			Keychata(portbuf);
		}
	}

	flg.key.keydata_set = 0;
	keydata_old = keydata;

}


/**
 *********************************************************************************************
 * @brief		keychataring
 *
 * @param		sw_t* const portbuf
 *
 * @return		none
 *********************************************************************************************
 */
static void Keychata(sw_t portdata)
{
	if (portbuf.all == portbuf_old.all)
	{
		if (key_chatacnt != 0)
		{
			if (--key_chatacnt == 0)
			{
				keydata_old = keydata;
				keydata.all = (portdata.all ^ 0xfc);
				flg.key.keydata_set = 1;
			}
		}
	}
	else
	{
		portbuf_old = portdata;
		key_chatacnt = cKEY_CHATA_CNT;
	}

}


/**
 *********************************************************************************************
 * @brief		user_setting
 *
 * @return		none
 *********************************************************************************************
 */
void user_setting(void)
{

	/* speed control */
	if(flg.motor.motor_change == 1)
	{
#if 0 // 20230616 ۍ [^[؂ւ̂߃RgAEg
		flg.motor.motor_change = 0;
#endif
		if(ctrl_motor == cMOTOR_CH0)
		{
			ctrl_motor = cMOTOR_CH1;
			speed_temp = speed_temp_ch1;
		}
		else
		{
			ctrl_motor = cMOTOR_CH0;
			speed_temp = speed_temp_ch0;
		}
		flg.motor.motor_action = 1;
		motor_action = cMOTOR_SPEED_NOCHG;
	}
	else
	{
		if(motor_action == cMOTOR_SPEED_STOP)
		{
			speed_temp = 0;
			speed_temp_ch0 = 0;
			speed_temp_ch1 = 0;
			pfc_state = cPFC_OFF;
			flg.motor.motor_action = 1;
		}
		else
		{
			if(motor_action == cMOTOR_SPEED_UP)
			{
				if(ctrl_motor == cMOTOR_CH0)
				{
					if(speed_temp_ch0 < cSPEED_USER_MAX)
					{
						speed_temp_ch0 += cSPEED_USER_UP_DOWN;
						if(speed_temp_ch0 >= cSPEED_USER_MAX)
						{
							speed_temp_ch0 = cSPEED_USER_MAX;
						}
						speed_temp = speed_temp_ch0;
						flg.motor.motor_action = 1;
					}
				}
				else if(ctrl_motor == cMOTOR_CH1)
				{
					if(speed_temp_ch1 < cSPEED_USER_MAX)
					{
						speed_temp_ch1 += cSPEED_USER_UP_DOWN;
						if(speed_temp_ch1 >= cSPEED_USER_MAX)
						{
							speed_temp_ch1 = cSPEED_USER_MAX;
						}
						speed_temp = speed_temp_ch1;
						flg.motor.motor_action = 1;
					}
				}
				else
				{
					/* no process */
				}
				motor_action = cMOTOR_SPEED_NOCHG;
			}
			else if(motor_action == cMOTOR_SPEED_DOWN)
			{
				if(ctrl_motor == cMOTOR_CH0)
				{
					if(speed_temp_ch0 >= cSPEED_USER_MIN)
					{
						speed_temp_ch0 -= cSPEED_USER_UP_DOWN;
						if(speed_temp_ch0 < cSPEED_USER_MIN)
						{
							speed_temp_ch0 = 0;
						}
						speed_temp = speed_temp_ch0;
						flg.motor.motor_action = 1;
					}
				}
				else if(ctrl_motor == cMOTOR_CH1)
				{
					if(speed_temp_ch1 >= cSPEED_USER_MIN)
					{
						speed_temp_ch1 -= cSPEED_USER_UP_DOWN;
						if(speed_temp_ch1 < cSPEED_USER_MIN)
						{
							speed_temp_ch1 = 0;
						}
						speed_temp = speed_temp_ch1;
						flg.motor.motor_action = 1;
					}
				}
				else
				{
					/* no process */
				}
				motor_action = cMOTOR_SPEED_NOCHG;
			}
			else
			{
				/* no process */
			}
		}
	}

	/* state control */
	if(flg.motor.motor_pfcchange == 1)
	{
#if 0 // 20230616 ۍ [^[؂ւ̂ߏύX
		flg.motor.motor_pfcchange = 0;
#endif
		pfc_state ^= 1;
		flg.motor.motor_action = 1;
	}
	else if(flg.disp.change_dac == 1)
	{
		flg.disp.change_dac = 0;
		
		if(dac_mode < cDAC_MDOE3)
		{
			dac_mode++;
		}
		else
		{
			dac_mode = cDAC_MDOE0;
		}
		
		flg.disp.change_disp_status = 1;
	}
	else if(flg.disp.change_ch == 1)
	{
		flg.disp.change_ch = 0;

		if(disp_ch < cDISP_CH2)
		{
			disp_ch++;
		}
		else
		{
			disp_ch = cDISP_CH0;
		}
		
		flg.disp.change_disp_status = 1;
		flg.key.keylongpush_done = 1;
	}
	else
	{
		/* no process */
	}
}

/**
 *********************************************************************************************
 * @brief		init_led
 *
 * @return		none
 *********************************************************************************************
 */
void init_led(void)
{
	/* Supply clock fsys to PORT */
	CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_PORTK, ENABLE);		/* LED1-3 */
	CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_PORTJ, ENABLE);		/* LED4 */
	CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_PORTV, ENABLE);		/* LED5-6 */
	
	TSB_PK_DATA_PK1 = cLED_OFF;									/* LED1 */
	TSB_PK_CR_PK1C = ENABLE;
	
	TSB_PK_DATA_PK2 = cLED_OFF;									/* LED2 */
	TSB_PK_CR_PK2C = ENABLE;
	
	TSB_PK_DATA_PK3 = cLED_OFF;									/* LED3 */
	TSB_PK_CR_PK3C = ENABLE;
	
	TSB_PJ_DATA_PJ5 = cLED_OFF;									/* LED4 */
	TSB_PJ_CR_PJ5C = ENABLE;
	
	TSB_PV_DATA_PV1 = cLED_OFF;									/* LED5 */
	TSB_PV_CR_PV1C = ENABLE;
	
	TSB_PV_DATA_PV0 = cLED_OFF;									/* LED6 */
	TSB_PV_CR_PV0C = ENABLE;
	

}


/**
 *********************************************************************************************
 * @brief		led_display
 *
 * @return		none
 *********************************************************************************************
 */
void led_display(void)
{

	/* LED lighting by UART communication */
	if(uart_mode == cUART_ERR)
	{
		LED_UART_ERR = led1_state;
	}
	else
	{
		LED_UART_ERR = cLED_OFF;
	}

	/* LED lighting by PFC status */
	if(pfc_state == cPFC_ON)
	{
		LED_PFC_ON = cLED_ON;
	}
	else
	{
		LED_PFC_ON = cLED_OFF;
	}

	/* LED lighting by Controlled Motor */
	if(ctrl_motor == cCONTROL_MOTOR_CH0)
	{
		LED_MOTOR_CH0 = cLED_ON;
	}
	else
	{
		LED_MOTOR_CH0 = cLED_OFF;
	}

	/* LED lighting by EMG status */
	if(flg.uart.emg_bit == 1)
	{
		if(control_ch[1] == cMOTOR_CH0)
		{
			LED_CH0_EMG = led4_state;
		}
		else
		{
			LED_CH0_EMG = cLED_OFF;
		}
		
		if(control_ch[1] == cMOTOR_CH1)
		{
			LED_CH1_EMG = led5_state;
		}
		else
		{
			LED_CH1_EMG = cLED_OFF;
		}
		
		if(control_ch[1] == cMOTOR_CH2)
		{
			LED_CH2_EMG = led6_state;
		}
		else
		{
			LED_CH2_EMG = cLED_OFF;
		}
	}
	else
	{
		LED_CH0_EMG = cLED_OFF;
		LED_CH1_EMG = cLED_OFF;
		LED_CH2_EMG = cLED_OFF;
	}
}

/**
 *********************************************************************************************
 * @brief		Initialize UART.
 *
 * @return		none
 *********************************************************************************************
 */
void init_uart(void)
{
	UART_InitTypeDef UART_DispInit;
	uint16_t	port_wait_counter;

	CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_UART0, ENABLE);
	CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_UART2, ENABLE);

	/* Setting UART0 port. */
	TSB_PC->FR1 |= (1 << 0U);
	TSB_PC->CR  |= (1 << 0U);
	TSB_PC->DATA|= (1 << 0U);	/* PC0:UT0TXDA */

	/* Setting UART2 port. */
	TSB_PU->FR1 |= ((1 << 0U) | (1 << 1U));
	TSB_PU->IE  |= (1 << 1U);	/* PU1:UT2RXD */
	TSB_PU->CR  |= (1 << 0U);
	TSB_PU->DATA|= (1 << 0U);	/* PU0:UT2TXDA */

	/* Wait for initialization of UART port */
	port_wait_counter = 0;
	while (port_wait_counter <= cPORT_WAIT_CNT)
	{
		SIWDT_WriteClearCode();
		port_wait_counter++;
	}

	UART_SWReset(UART0);
	UART_SWReset(UART2);

	/*initialize the UART0 struct */
	UART_DispInit.BaudRate = 115200U;
	UART_DispInit.DataBits = UART_DATA_BITS_8;
	UART_DispInit.DataInvert = UART_NO_DATA_INVERT;
	UART_DispInit.TransferDirection = UART_LSB;
	UART_DispInit.StopBits = UART_STOP_BITS_1;
	UART_DispInit.Parity = UART_NO_PARITY;
	UART_DispInit.Mode = (UART_ENABLE_TX);
	UART_DispInit.FlowCtrl = UART_NONE_FLOW_CTRL;

	UART_TxINTCtrl(UART0,ENABLE);

	UART_Init(UART0, &UART_DispInit);

	NVIC_SetPriority(INTSC0TX_IRQn, INT_UART_LEVEL); 			/* Interruption level set */
	NVIC_ClearPendingIRQ(INTSC0TX_IRQn);
	NVIC_EnableIRQ(INTSC0TX_IRQn);							/* INTTX enable */

	/*initialize the UART2 struct */
	UART_DispInit.BaudRate = 9600U;//115200U;
	UART_DispInit.DataBits = UART_DATA_BITS_8;
	UART_DispInit.DataInvert = UART_NO_DATA_INVERT;
	UART_DispInit.TransferDirection = UART_LSB;
	UART_DispInit.StopBits = UART_STOP_BITS_1;
	UART_DispInit.Parity = UART_NO_PARITY;
	UART_DispInit.Mode = (UART_ENABLE_TX | UART_ENABLE_RX);
	UART_DispInit.FlowCtrl = UART_NONE_FLOW_CTRL;

	UART_TxINTCtrl(UART2,ENABLE);
	UART_RxINTCtrl(UART2,ENABLE);

	UART_Init(UART2, &UART_DispInit);

	NVIC_SetPriority(INTSC2TX_IRQn, INT_UART_LEVEL); 			/* Interruption level set */
	NVIC_SetPriority(INTSC2RX_IRQn, INT_UART_LEVEL); 			/* Interruption level set */
	NVIC_ClearPendingIRQ(INTSC2TX_IRQn);
	NVIC_ClearPendingIRQ(INTSC2RX_IRQn);
	NVIC_EnableIRQ(INTSC2TX_IRQn);							/* INTTX enable */
	NVIC_EnableIRQ(INTSC2RX_IRQn);							/* INTRX enable */

}


/**
 *********************************************************************************************
 * @brief		uart control.
 *
 * @return		none
 *********************************************************************************************
 */
void uart_control(void)
{
	switch(uart_mode)
	{
		case cUART_START_COMMAND:
			uart_mode = cUART_RECEIVE_WAIT;
			send_mode = cSEND_START_COMMAND;
			rxpnt = 0;
			command_id = cREQ_SYSTEM_START;
			send_data_set(command_id);
			uart_send_start();
			break;

		case cUART_SEND_WAIT:
			if(flg.disp.get_status_1 == 1)
			{
				flg.disp.get_status_1 = 0;
				uart_mode = cUART_RECEIVE_WAIT;
				send_mode = cSEND_STATUS_DISP;
				rxpnt = 0;
				command_id = cGET_STATE_EMG;
				send_data_set(command_id);
				uart_send_start();
				uart_err_count = 0;
			}
			else if(flg.disp.get_status_2 == 1)
			{
				flg.disp.get_status_2 = 0;
				uart_mode = cUART_RECEIVE_WAIT;
				send_mode = cSEND_ROTATE_DISP;
				rxpnt = 0;
				if(disp_ch != cDISP_CH2)
				{
					command_id = cGET_MOTOR_SPEED;
				}
				else
				{
					command_id = cGET_VDC_VOLTAGE;
				}
				send_data_set(command_id);
				uart_send_start();
				uart_err_count = 0;
			}
			else if(flg.motor.motor_action == 1)
			{
				flg.motor.motor_action = 0;
				uart_mode = cUART_RECEIVE_WAIT;
				send_mode = cSEND_ROTATE_SPEED;
				rxpnt = 0;
				if(motor_action != cMOTOR_SPEED_STOP)
				{
#if 0 // 20230616 ۍ [^[؂ւ̂ߏύX
					command_id = cREQ_ROTATE_MOTOR;
#else
					if(flg.motor.motor_pfcchange == 1)
					{
						flg.motor.motor_pfcchange = 0;
						command_id = cREQ_ON_OFF_PFC;
					}
					else if(flg.motor.motor_change == 1)
					{
						flg.motor.motor_change = 0;
						command_id = cREQ_CANGE_MOTOR;
					}
					else
					{
						command_id = cREQ_ROTATE_MOTOR;
					}
#endif
					send_data_set(command_id);
					uart_send_start();
					uart_err_count = 0;
				}
				else
				{
					motor_action = cMOTOR_SPEED_NOCHG;
					command_id = cREQ_ALL_STOP_MOTOR;
					send_data_set(command_id);
					uart_send_start();
					uart_err_count = 0;
				}
			}
			else if(flg.disp.change_disp_status == 1)
			{
				flg.disp.change_disp_status = 0;
				uart_mode = cUART_RECEIVE_WAIT;
				send_mode = cSEND_CHANGE_DISP;
				rxpnt = 0;
				command_id = cREQ_STATUS_CH;
				send_data_set(command_id);
				uart_send_start();
				uart_err_count = 0;
			}
			else
			{
				/* no process */
			}
			
			timeout_timer = 0;
			flg.timer.timeout = 0;
			break;

		case cUART_RECEIVE_WAIT:
			if(flg.uart.data_receive == 1)
			{
				flg.uart.data_receive = 0;
				timeout_timer = 0;

				if(data_check() == cRESULT_NG)
				{
					/* w/t error */
					if(++uart_err_count < 10)
					{
						rxpnt = 0;
						send_data_set(command_id);
						uart_send_start();
					}
					else
					{
						uart_send_stop();
						uart_mode = cUART_ERR;
					}
				}
				else
				{
					/* No error */

					uart_err_count = 0;

					switch(send_mode)
					{
						case cSEND_START_COMMAND:
							send_mode = cSEND_IDLE;
							uart_mode = cUART_SEND_WAIT;
							break;

						case cSEND_STATUS_DISP:
							if(command_id >= cGET_HVMOS_TEMP)
							{
								flg.disp.status_disp = 1;
								send_mode = cSEND_IDLE;
								uart_mode = cUART_SEND_WAIT;
							}
							else
							{
								if(command_id == cGET_MODULATION)
								{
									rxpnt = 0;
									command_id = cGET_OUTPIPE_TEMP;
									send_data_set(command_id);
									uart_send_start();
								}
								else
								{
									rxpnt = 0;
									command_id++;
									send_data_set(command_id);
									uart_send_start();
								}
							}
							break;

						case cSEND_ROTATE_DISP:
							flg.disp.speed_disp = 1;
							send_mode = cSEND_IDLE;
							uart_mode = cUART_SEND_WAIT;
							break;

						case cSEND_ROTATE_SPEED:
#if 0 // 20230616 ۍ [^[؂ւ̂ߏύX
							if((command_id >= cREQ_ON_OFF_PFC) || (command_id >= cREQ_ALL_STOP_MOTOR))
#else
							if(command_id != cREQ_CANGE_MOTOR)
#endif
							{
								send_mode = cSEND_IDLE;
								uart_mode = cUART_SEND_WAIT;
							}
							else
							{
								rxpnt = 0;
#if 0 // 20230616 ۍ [^[؂ւ̂ߏύX
								command_id++;
#else
								command_id = cREQ_ROTATE_MOTOR;
#endif
								send_data_set(command_id);
								uart_send_start();
							}
							break;

						case cSEND_CHANGE_DISP:
							if(command_id >= cREQ_STATUS_DAC)
							{
								flg.disp.get_status_1 = 1;
								uart_mode = cUART_SEND_WAIT;
							}
							else
							{
								rxpnt = 0;
								command_id++;
								send_data_set(command_id);
								uart_send_start();
							}
							break;

						case cSEND_IDLE:
						default:
							break;
					}
				}
			}
			else
			{
				if(flg.timer.timeout == 1)
				{
					flg.timer.timeout = 0;
					if(send_mode == cSEND_START_COMMAND)
					{
						rxpnt = 0;
						command_id = cREQ_SYSTEM_START;
						send_data_set(command_id);
						uart_send_start();
					}
					else
					{
						if(++uart_err_count < 10)
						{
							send_data_set(command_id);
							uart_send_start();
						}
						else
						{
							uart_send_stop();
							uart_mode = cUART_ERR;
						}
					}
				}
			}
			break;

		case cUART_ERR:
		default:
			break;
	}

}


/**
 *********************************************************************************************
 * @brief		send data setting.
 *
 * @return		none
 *********************************************************************************************
 */
static void send_data_set(uint8_t id)
{
	uint8_t i;

	for(i= 0; i< 6; i++)
	{
		sendbuffer[i] = 0x00;
	}

	sendbuffer[0] = id;

	if(id == cREQ_ROTATE_MOTOR)
	{
		sendbuffer[1] = speed_temp;
	}
	if(id == cREQ_CANGE_MOTOR)
	{
		sendbuffer[1] = ctrl_motor;
	}
	if(id == cREQ_ON_OFF_PFC)
	{
		sendbuffer[1] = pfc_state;
	}
	if(id == cREQ_ALL_STOP_MOTOR)
	{
		sendbuffer[1] = 0x00;
	}
	if(id == cREQ_STATUS_CH)
	{
		sendbuffer[1] = disp_ch;
	}
	if(id == cREQ_STATUS_DAC)
	{
		sendbuffer[1] = dac_mode;
	}

	sendbuffer[5] = 0;
	for(i= 0; i< 5; i++)
	{
		sendbuffer[5] += sendbuffer[i];
	}

}


/**
 *********************************************************************************************
 * @brief		data check.
 *
 * @return		uint8_t result
 *********************************************************************************************
 */
static uint8_t data_check(void)
{
	uint8_t i;
	uint8_t datacheck_result = cRESULT_OK;
	uint32_t receive_value;
	flg.uart.emg_bit = 0;
	flg.uart.na_bit = 0;
	flg.uart.ack_bit = 0;
	receive_checksum = 0;

	if((receivedata[1] & cEMGBIT_MASK) == cEMGBIT_MASK)
	{
		flg.uart.emg_bit = 1;
	}

	if((receivedata[1] & cNABIT_MASK) == cNABIT_MASK)
	{
		flg.uart.na_bit = 1;
	}

	if((receivedata[1] & cACKBIT_MASK) == cACKBIT_MASK)
	{
		flg.uart.ack_bit = 1;
	}

	for(i= 0; i< 6; i++)
	{
		receive_checksum += receivedata[i];
	}

	if(receivedata[6] != receive_checksum)
	{
		flg.uart.ack_bit = 1;
	}

	if(flg.uart.ack_bit == 0){
		switch(receivedata[0])
		{
			case cREQ_SYSTEM_START:
				flg.uart.initialize = 1;
				flg.disp.get_status_1 = 1;
				break;

			case cGET_STATE_EMG:
				state_emg[0] = receivedata[2];
				break;

			case cGET_STAGE:
				main_stage[0] = receivedata[2];
				break;

			case cGET_CONTROL_CH:
				control_ch[0] = receivedata[2];
				break;

			case cGET_CARRIER_FREQUENCY:
				receive_value  = (uint32_t)receivedata[2];
				receive_value |= (uint32_t)(receivedata[3] << 8);
				receive_value |= (uint32_t)(receivedata[4] << 16);
				receive_value |= (uint32_t)(receivedata[5] << 24);

				carrier_frequency[0] = receive_value;
				break;

			case cGET_MOTOR_SPEED_MIN:
				receive_value  = (uint32_t)receivedata[2];
				receive_value |= (uint32_t)(receivedata[3] << 8);
				receive_value |= (uint32_t)(receivedata[4] << 16);
				receive_value |= (uint32_t)(receivedata[5] << 24);

				motor_speed_min[0] = receive_value;
				break;

			case cGET_MOTOR_SPEED_MAX:
				receive_value  = (uint32_t)receivedata[2];
				receive_value |= (uint32_t)(receivedata[3] << 8);
				receive_value |= (uint32_t)(receivedata[4] << 16);
				receive_value |= (uint32_t)(receivedata[5] << 24);

				motor_speed_max[0] = receive_value;
				break;

			case cGET_DEAD_TIME:
				receive_value  = (uint32_t)receivedata[2];
				receive_value |= (uint32_t)(receivedata[3] << 8);
				receive_value |= (uint32_t)(receivedata[4] << 16);
				receive_value |= (uint32_t)(receivedata[5] << 24);

				dead_time[0] = ((float32_t)receive_value / 100);
				break;

			case cGET_GATE_ACTIVE:
				gate_active[0] = receivedata[2];
				break;

			case cGET_POSITION_DITECT:
				position_detect[0] = receivedata[2];
				break;

			case cGET_VDC_VOLTAGE:
				receive_value  = (uint32_t)receivedata[2];
				receive_value |= (uint32_t)(receivedata[3] << 8);
				receive_value |= (uint32_t)(receivedata[4] << 16);
				receive_value |= (uint32_t)(receivedata[5] << 24);

				vdc_voltage[0] = ((float32_t)receive_value / 100);
				break;

			case cGET_OUTROOM_TEMP:
				receive_value  = (uint32_t)receivedata[2];
				receive_value |= (uint32_t)(receivedata[3] << 8);
				receive_value |= (uint32_t)(receivedata[4] << 16);
				receive_value |= (uint32_t)(receivedata[5] << 24);

				outroom_temp[0] = flg.uart.na_bit;
				outroom_temp[1] = ((float32_t)receive_value / 100);
				break;

			case cGET_OUTPIPE_TEMP:
				receive_value  = (uint32_t)receivedata[2];
				receive_value |= (uint32_t)(receivedata[3] << 8);
				receive_value |= (uint32_t)(receivedata[4] << 16);
				receive_value |= (uint32_t)(receivedata[5] << 24);

				outpipe_temp[0] = flg.uart.na_bit;
				outpipe_temp[1] = ((float32_t)receive_value / 100);
				break;

			case cGET_EXHAUST_TEMP:
				receive_value  = (uint32_t)receivedata[2];
				receive_value |= (uint32_t)(receivedata[3] << 8);
				receive_value |= (uint32_t)(receivedata[4] << 16);
				receive_value |= (uint32_t)(receivedata[5] << 24);

				exhaust_temp[0] = flg.uart.na_bit;
				exhaust_temp[1] = ((float32_t)receive_value / 100);
				break;

			case cGET_DIODE_TEMP:
				receive_value  = (uint32_t)receivedata[2];
				receive_value |= (uint32_t)(receivedata[3] << 8);
				receive_value |= (uint32_t)(receivedata[4] << 16);
				receive_value |= (uint32_t)(receivedata[5] << 24);

				diode_temp[0] = flg.uart.na_bit;
				diode_temp[1] = ((float32_t)receive_value / 100);
				break;

			case cGET_IGBT_TEMP:
				receive_value  = (uint32_t)receivedata[2];
				receive_value |= (uint32_t)(receivedata[3] << 8);
				receive_value |= (uint32_t)(receivedata[4] << 16);
				receive_value |= (uint32_t)(receivedata[5] << 24);

				igbt_temp[0] = flg.uart.na_bit;
				igbt_temp[1] = ((float32_t)receive_value / 100);
				break;

			case cGET_HVMOS_TEMP:
				receive_value  = (uint32_t)receivedata[2];
				receive_value |= (uint32_t)(receivedata[3] << 8);
				receive_value |= (uint32_t)(receivedata[4] << 16);
				receive_value |= (uint32_t)(receivedata[5] << 24);

				hvmos_temp[0] = flg.uart.na_bit;
				hvmos_temp[1] = ((float32_t)receive_value / 100);
				break;

			case cGET_U_VOLTAGE:
				receive_value  = (uint32_t)receivedata[2];
				receive_value |= (uint32_t)(receivedata[3] << 8);
				receive_value |= (uint32_t)(receivedata[4] << 16);
				receive_value |= (uint32_t)(receivedata[5] << 24);

				u_voltage[0] = ((float32_t)receive_value / 100);
				break;

			case cGET_V_VOLTAGE:
				receive_value  = (uint32_t)receivedata[2];
				receive_value |= (uint32_t)(receivedata[3] << 8);
				receive_value |= (uint32_t)(receivedata[4] << 16);
				receive_value |= (uint32_t)(receivedata[5] << 24);

				v_voltage[0] = ((float32_t)receive_value / 100);
				break;

			case cGET_W_VOLTAGE:
				receive_value  = (uint32_t)receivedata[2];
				receive_value |= (uint32_t)(receivedata[3] << 8);
				receive_value |= (uint32_t)(receivedata[4] << 16);
				receive_value |= (uint32_t)(receivedata[5] << 24);

				w_voltage[0] = ((float32_t)receive_value / 100);
				break;

			case cGET_DAC_DATA:
				dac_data[0] = receivedata[2];
				dac_data[1] = receivedata[3];
				dac_data[2] = receivedata[4];
				dac_data[3] = receivedata[5];
				break;

			case cGET_INTERNAL_AMP:
				internal_amp[0] = receivedata[2];
				break;

			case cGET_DIRECTION:
				direction[0] = receivedata[2];
				break;

			case cGET_MODULATION:
				modulation[0] = receivedata[2];
				break;

			case cGET_MOTOR_SPEED:
				motor_speed[0] = (int8_t)receivedata[2];
				break;

			default:
				break;
		}
	}
	else {
		datacheck_result = cRESULT_NG;
	}

	return datacheck_result;

}


/**
 *********************************************************************************************
 * @brief		uart send start.
 *
 * @return		none
 *********************************************************************************************
 */
static void uart_send_start(void)
{
	txpnt = 0;
	UART_SetTxData(UART2,sendbuffer[0]);
}


/**
 *********************************************************************************************
 * @brief		uart send stop.
 *
 * @return		none
 *********************************************************************************************
 */
static void uart_send_stop(void)
{
	UART_TxINTCtrl(UART2, DISABLE);
}


/**
 *********************************************************************************************
 * @brief		terminal display.
 *
 * @return		none
 *********************************************************************************************
 */
const char tDAC_DATA[11][18] = {
	"TMPREG0",
	"TMPREG1",
	"TMPREG2",
	"theta.half[1]",
	"Id_ref",
	"Id",
	"Iq_ref",
	"Iq",
	"omega_com.half[1]",
	"omega.half[1]",
	"omega_dev"
};

void terminal_display(void)
{
	switch(disp_mode)
	{
		case cDISP_IDLE:
			if(flg.disp.status_disp == 1)
			{
				terminaldisp_dataset();
				disp_mode = cDISP_STATUS;
				flg.disp.status_disp = 0;
			}
			else if(flg.disp.speed_disp == 1)
			{
				if(control_ch[1] != cMOTOR_CH2)
				{
					if((motor_speed[0] > 0) && (flg.uart.emg_bit == 0)){
						flg.disp.disp_stop = 0;
					}
				}
				else
				{
					if((pfc_state != cPFC_OFF) && (flg.uart.emg_bit == 0)){
						flg.disp.disp_stop = 0;
					}
				}
				terminaldisp_dataset();
				disp_mode = cDISP_ROTATE_SPEED;
				flg.disp.speed_disp = 0;
			}
			else
			{
				/* no process */
			}
			break;

		case cDISP_STATUS:
			switch(disp_ch)
			{
				case cDISP_CH0:
					switch(disp_id)
					{
						case cDISP_CONTROL_CH:
							printf("Control CH = CH%d\n", control_ch[1]);
							break;

						case cDISP_CARRIER_FREQUENCY:
							printf("Carrier Frequency = %dHz\n", carrier_frequency[1]);
							break;

						case cDISP_MOTOR_SPEED_MIN_MAX:
							printf("Hz MIN/MAX = %d/%d\n", motor_speed_min[1], motor_speed_max[1]);
							break;

						case cDISP_DEAD_TIME:
							printf("Dead Time = %-2.1fus\n", dead_time[1]);
							break;

						case cDISP_GATE_ACTIVE:
							if(gate_active[1] == cGATE_ACTIVE_H_H)
							{
								printf("Gate Active = H/H\n");
							}
							else if(gate_active[1] == cGATE_ACTIVE_L_L)
							{
								printf("Gate Active = L/L\n");
							}
							else if(gate_active[1] == cGATE_ACTIVE_H_L)
							{
								printf("Gate Active = H/L\n");
							}
							else
							{
								printf("Gate Active = L/H\n");
							}
							break;

						case cDISP_POSITION_DITECT:
							if(position_detect[1] == cPOSITION_DETECT_3SHUNT)
							{
								printf("Position Detect = 3Shunt\n");
							}
							else
							{
								printf("Position Detect = 1Shunt(SPWM On/Off)\n");
							}
							break;

						case cDISP_VDC_VOLTAGE:
							printf("VDC Voltage = %-4.1fV\n", vdc_voltage[1]);
							break;

						case cDISP_OUTROOM_TEMP:
							if(outroom_temp[0] == 1)
							{
								printf("OUTROOM Temp = N.A\n");
							}
							else
							{
								printf("OUTROOM Temp = %-4.1fdegree\n", outroom_temp[2]);
							}
							break;

						case cDISP_OUTPIPE_TEMP:
							if(outpipe_temp[0] == 1)
							{
								printf("OUTPIPE Temp = N.A\n");
							}
							else
							{
								printf("OUTPIPE Temp = %-4.1fdegree\n", outpipe_temp[2]);
							}
							break;

						case cDISP_EXHAUST_TEMP:
							if(exhaust_temp[0] == 1)
							{
								printf("EXHAUST Temp = N.A\n");
							}
							else
							{
								printf("EXHAUST Temp = %-4.1fdegree\n", exhaust_temp[2]);
							}
							break;

						case cDISP_DIODE_TEMP:
							if(diode_temp[0] == 1)
							{
								printf("DIODE Temp = N.A\n");
							}
							else
							{
								printf("DIODE Temp = %-4.1fdegree\n", diode_temp[2]);
							}
							break;

						case cDISP_IGBT_TEMP:
							if(igbt_temp[0] == 1)
							{
								printf("IGBT Temp = N.A\n");
							}
							else
							{
								printf("IGBT Temp = %-4.1fdegree\n", igbt_temp[2]);
							}
							break;

						case cDISP_HVMOS_TEMP:
							if(hvmos_temp[0] == 1)
							{
								printf("HVMOS Temp = N.A\n");
							}
							else
							{
								printf("HVMOS Temp = %-4.1fdegree\n", hvmos_temp[2]);
							}
							break;

						case cDISP_UVW_VOLTAGE:
							printf("UVW 0A Voltage = %-3.2fV/%-3.2fV/%-3.2fV\n", u_voltage[1], v_voltage[1], w_voltage[1]);
							break;

						case cDISP_DAC_DATA:
							printf("DAC DATA = A:%s B:%s C:%s D:%s\n", tDAC_DATA[dac_data[4]], tDAC_DATA[dac_data[5]],tDAC_DATA[dac_data[6]],tDAC_DATA[dac_data[7]]);
							break;

						case cDISP_INTERNAL_AMP:
							if(internal_amp[1] == cINTERNAL_AMP_NO)
							{
								printf("Internal AMP = NO\n");
							}
							else
							{
								printf("Internal AMP = YES\n");
							}
							break;

						case cDISP_DIRECTION:
							if(direction[1] == cDIRECTION_CW)
							{
								printf("Direction = CW\n");
							}
							else
							{
								printf("Direction = CCW\n");
							}
							break;

						case cDISP_MODULATION:
							if(modulation[1] == cMODULATION_3PHASE)
							{
								printf("Modulation = 3Phase\n");
							}
							else
							{
								printf("Modulation = 2Phase\n");
							}
							break;

						default:
							break;
					}

					if(++disp_id > cDISP_MODULATION)
					{
						disp_mode = cDISP_IDLE;
						disp_id = cDISP_CONTROL_CH;
					}
					break;

				case cDISP_CH1:
					switch(disp_id)
					{
						case cDISP_CONTROL_CH:
							printf("Control CH = CH%d\n", control_ch[1]);
							break;

						case cDISP_CARRIER_FREQUENCY:
							printf("Carrier Frequency = %dHz\n", carrier_frequency[1]);
							break;

						case cDISP_MOTOR_SPEED_MIN_MAX:
							printf("Hz MIN/MAX = %d/%d\n", motor_speed_min[1], motor_speed_max[1]);
							break;

						case cDISP_DEAD_TIME:
							printf("Dead Time = %-2.1fus\n", dead_time[1]);
							break;

						case cDISP_GATE_ACTIVE:
							if(gate_active[1] == cGATE_ACTIVE_H_H)
							{
								printf("Gate Active = H/H\n");
							}
							else if(gate_active[1] == cGATE_ACTIVE_L_L)
							{
								printf("Gate Active = L/L\n");
							}
							else if(gate_active[1] == cGATE_ACTIVE_H_L)
							{
								printf("Gate Active = H/L\n");
							}
							else
							{
								printf("Gate Active = L/H\n");
							}
							break;

						case cDISP_POSITION_DITECT:
							if(position_detect[1] == cPOSITION_DETECT_3SHUNT)
							{
								printf("Position Detect = 3Shunt\n");
							}
							else
							{
								printf("Position Detect = 1Shunt(SPWM On/Off)\n");
							}
							break;

						case cDISP_VDC_VOLTAGE:
							printf("VDC Voltage = %-4.1fV\n", vdc_voltage[1]);
							break;

						case cDISP_OUTROOM_TEMP:
							if(outroom_temp[0] == 1)
							{
								printf("OUTROOM Temp = N.A\n");
							}
							else
							{
								printf("OUTROOM Temp = %-4.1fdegree\n", outroom_temp[2]);
							}
							break;

						case cDISP_OUTPIPE_TEMP:
							if(outpipe_temp[0] == 1)
							{
								printf("OUTPIPE Temp = N.A\n");
							}
							else
							{
								printf("OUTPIPE Temp = %-4.1fdegree\n", outpipe_temp[2]);
							}
							break;

						case cDISP_EXHAUST_TEMP:
							if(exhaust_temp[0] == 1)
							{
								printf("EXHAUST Temp = N.A\n");
							}
							else
							{
								printf("EXHAUST Temp = %-4.1fdegree\n", exhaust_temp[2]);
							}
							break;

						case cDISP_DIODE_TEMP:
							if(diode_temp[0] == 1)
							{
								printf("DIODE Temp = N.A\n");
							}
							else
							{
								printf("DIODE Temp = %-4.1fdegree\n", diode_temp[2]);
							}
							break;

						case cDISP_IGBT_TEMP:
							if(igbt_temp[0] == 1)
							{
								printf("IGBT Temp = N.A\n");
							}
							else
							{
								printf("IGBT Temp = %-4.1fdegree\n", igbt_temp[2]);
							}
							break;

						case cDISP_HVMOS_TEMP:
							if(hvmos_temp[0] == 1)
							{
								printf("HVMOS Temp = N.A\n");
							}
							else
							{
								printf("HVMOS Temp = %-4.1fdegree\n", hvmos_temp[2]);
							}
							break;

						case cDISP_UVW_VOLTAGE:
							printf("UVW 0A Voltage = %-3.2fV/%-3.2fV/%-3.2fV\n", u_voltage[1], v_voltage[1], w_voltage[1]);
							break;

						case cDISP_DAC_DATA:
							printf("DAC DATA = A:%s B:%s C:%s D:%s\n", tDAC_DATA[dac_data[4]], tDAC_DATA[dac_data[5]],tDAC_DATA[dac_data[6]],tDAC_DATA[dac_data[7]]);
							break;

						case cDISP_INTERNAL_AMP:
							if(internal_amp[1] == cINTERNAL_AMP_NO)
							{
								printf("Internal AMP = NO\n");
							}
							else
							{
								printf("Internal AMP = YES\n");
							}
							break;

						case cDISP_DIRECTION:
							if(direction[1] == cDIRECTION_CW)
							{
								printf("Direction = CW\n");
							}
							else
							{
								printf("Direction = CCW\n");
							}
							break;

						case cDISP_MODULATION:
							if(modulation[1] == cMODULATION_3PHASE)
							{
								printf("Modulation = 3Phase\n");
							}
							else
							{
								printf("Modulation = 2Phase\n");
							}
							break;

						default:
							break;
					}

					if(++disp_id > cDISP_MODULATION)
					{
						disp_mode = cDISP_IDLE;
						disp_id = cDISP_CONTROL_CH;
					}
					break;

				case cDISP_CH2:
					switch(disp_id)
					{
						case cDISP_CONTROL_CH:
							if(pfc_state == cPFC_ON)
							{
								printf("PFC On/Off = On\n");
							}
							else
							{
								printf("PFC On/Off = Off\n");
							}

							printf("Control CH = CH%d\n", control_ch[1]);
							break;

						case cDISP_CARRIER_FREQUENCY:
							printf("Carrier Frequency = %dHz\n", carrier_frequency[1]);
							break;

						case cDISP_GATE_ACTIVE:
							if(gate_active[1] == cGATE_ACTIVE_H_H)
							{
								printf("Gate Active = H/H\n");
							}
							else if(gate_active[1] == cGATE_ACTIVE_L_L)
							{
								printf("Gate Active = L/L\n");
							}
							else if(gate_active[1] == cGATE_ACTIVE_H_L)
							{
								printf("Gate Active = H/L\n");
							}
							else
							{
								printf("Gate Active = L/H\n");
							}
							break;

						case cDISP_POSITION_DITECT:
							if(position_detect[1] == cPOSITION_DETECT_3SHUNT)
							{
								printf("Position Detect = 3Shunt\n");
							}
							else
							{
								printf("Position Detect = 1Shunt\n");
							}
							break;

						case cDISP_VDC_VOLTAGE:
							printf("VDC Voltage = %-4.1fV\n", vdc_voltage[1]);
							break;

						case cDISP_UVW_VOLTAGE:
							printf("UVW 0A Voltage = %-3.2fV/%-3.2fV/%-3.2fV\n", u_voltage[1], v_voltage[1], w_voltage[1]);
							break;

						case cDISP_DAC_DATA:
							printf("DAC DATA = A:%s B:%s C:%s D:%s\n", tDAC_DATA[dac_data[4]], tDAC_DATA[dac_data[5]],tDAC_DATA[dac_data[6]],tDAC_DATA[dac_data[7]]);
							break;

						case cDISP_INTERNAL_AMP:
							if(internal_amp[1] == cINTERNAL_AMP_NO)
							{
								printf("Internal AMP = NO\n");
							}
							else
							{
								printf("Internal AMP = YES\n");
							}
							break;

						default:
							break;
					}
					if(++disp_id > cDISP_INTERNAL_AMP)
					{
						disp_mode = cDISP_IDLE;
						disp_id = cDISP_CONTROL_CH;
					}
					break;

				default:
					break;
			}
			break;

		case cDISP_ROTATE_SPEED:
			if(flg.disp.disp_stop == 0)
			{
				if(flg.uart.emg_bit == 1)
				{
					printf("CH%d : EMG_S\n", control_ch[1]);
					flg.disp.disp_stop = 1;
				}
				else if((motor_speed[1] == 0) && (control_ch[1] != cMOTOR_CH2))
				{
					/* printf֐ŕ\ */
					printf("CH%d = 0Hz\n", control_ch[1]);
					flg.disp.disp_stop = 1;
				}
				else if((pfc_state == cPFC_OFF) && (control_ch[1] == cMOTOR_CH2))
				{
					printf("CH%d = %-4.1fV\n", control_ch[1], 0.0);
					flg.disp.disp_stop = 1;
				}
				else
				{
					if(control_ch[1] != cMOTOR_CH2)
					{
						printf("CH%d = %dHz\n", control_ch[1], motor_speed[1]);
					}
					else
					{
						printf("CH%d = %-4.1fV\n", control_ch[1], vdc_voltage[1]);
					}
				}
			}
			disp_mode = cDISP_IDLE;
			break;

		default:
			break;
	}

}


/**
 *********************************************************************************************
 * @brief		terminal display data setting.
 *
 * @return		none
 *********************************************************************************************
 */
static void terminaldisp_dataset(void)
{
	if(flg.disp.status_disp == 1)
	{
		/* GET_STATE_EMG */
		state_emg[1] = state_emg[0];

		/* GET_STATE */
		main_stage[1] = main_stage[0];

		/* GET_CONTROL_CH */
		control_ch[1] = control_ch[0];

		/* GET_CARRIER_FREQUENCY */
		carrier_frequency[1] = carrier_frequency[0];

		/* GET_MOTOR_SPEED_MIN */
		motor_speed_min[1] = motor_speed_min[0];

		/* GET_MOTOR_SPEED_MAX */
		motor_speed_max[1] = motor_speed_max[0];

		/* GET_DEAD_TIME */
		dead_time[1] = dead_time[0];

		/* GET_GATE_ACTIVE */
		gate_active[1] = gate_active[0];

		/* GET_POSITION_DITECT */
		position_detect[1] = position_detect[0];

		/* GET_VDC_VOLTAGE */
		vdc_voltage[1] = vdc_voltage[0];

		/* GET_OUTROOM_TEMP */
		outroom_temp[2] = outroom_temp[1];

		/* GET_OUTPIPE_TEMP */
		outpipe_temp[2] = outpipe_temp[1];

		/* GET_EXHAUST_TEMP */
		exhaust_temp[2] = exhaust_temp[1];

		/* GET_DIODE_TEMP */
		diode_temp[2] = diode_temp[1];

		/* GET_IGBT_TEMP */
		igbt_temp[2] = igbt_temp[1];

		/* GET_HVMOS_TEMP */
		hvmos_temp[2] = hvmos_temp[1];

		/* GET_U_VOLTAGE */
		u_voltage[1] = u_voltage[0];

		/* GET_V_VOLTAGE */
		v_voltage[1] = v_voltage[0];

		/* GET_W_VOLTAGE */
		w_voltage[1] = w_voltage[0];

		/* GET_DAC_DATA */
		dac_data[4] = dac_data[0];
		dac_data[5] = dac_data[1];
		dac_data[6] = dac_data[2];
		dac_data[7] = dac_data[3];

		/* GET_INTERNAL_AMP */
		internal_amp[1] = internal_amp[0];

		/* GET_DIRECTION */
		direction[1] = direction[0];

		/* GET_MODULATIO */
		modulation[1] = modulation[0];
	}
	else if(flg.disp.speed_disp == 1)
	{
		/* GET_MOTOR_SPEED */
		motor_speed[1] = motor_speed[0];

		/* GET_VDC_VOLTAGE */
		vdc_voltage[1] = vdc_voltage[0];
	}
	else
	{
		/* no process */
	}

}


/**
 *********************************************************************************************
 * @brief		Transmission completion interrupt.
 *
 * @return		none
 *********************************************************************************************
 */
void INT_uart2tx(void)
{
	if(UART_GetErrState(UART2) == UART_NO_ERR)
	{
		if(++txpnt >= cSEND_DATA_NUM)
		{
			txpnt = 0;
			UART_ClearTransferStatus(UART2, UART_TRANSMIT);
		}
		else
		{
			UART_SetTxData(UART2,sendbuffer[txpnt]);
		}
	}

}


/**
 *********************************************************************************************
 * @brief		Receive completion interrupt.
 *
 * @return		none
 *********************************************************************************************
 */
void INT_uart2rx(void)
{
	uint8_t	dat;

	dat = UART_GetRxData(UART2);

	if(UART_GetErrState(UART2) == UART_NO_ERR)
	{
		receivebuffer[rxpnt] = dat;

		if(++rxpnt >= cRECEIVE_DATA_NUM)
		{
			receivedata[0] = receivebuffer[0];
			receivedata[1] = receivebuffer[1];
			receivedata[2] = receivebuffer[2];
			receivedata[3] = receivebuffer[3];
			receivedata[4] = receivebuffer[4];
			receivedata[5] = receivebuffer[5];
			receivedata[6] = receivebuffer[6];

			flg.uart.data_receive = 1;
			rxpnt = 0;

			UART_ClearTransferStatus(UART2, UART_RECEPTION);
		}
	}
}


/**
 *********************************************************************************************
 * @brief		send data to UART.
 *
 * @return		none
 *********************************************************************************************
 */
static int send_char(uint8_t sendData)
{
	int retStatus = (int)sendData;
	UART_SetTxData(UART0,sendData);

	while(UART_GetTransferStatus(UART0, UART_TRANSMIT) == 0)
	{															/* Check finish of send			*/
	}

	UART_ClearTransferStatus(UART0, UART_TRANSMIT);
    

	return retStatus;
}


/*------------------------------------------------------------------------------*/
/*  Semihosting Retarget                                                        */
/*------------------------------------------------------------------------------*/
#if defined ( __ARMCC_VERSION   )      /* RealView Compiler */
FILE __stdout;
FILE __stdin;
int fputc(int ch, FILE *f) {
#elif defined ( __ICCARM__ )    /*IAR Compiler */
int putchar(int ch) {
#endif
    return (send_char((uint8_t)ch));
}
/******************************** END OF FILE *********************************/
